
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 2000, 2001 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * transform.c
 *
 * Copyright (C) 2000, 2001 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: Sky dome creator for RW3.
 *
 ****************************************************************************/

#include "rwcore.h"
#include "rpworld.h"

#include "skeleton.h"

#include "main.h"
#include "skycolor.h"

RwReal CameraElevation = 25.0f;
RwReal CameraAzimuth = 345.0f;

RwV3d ClumpSpaceCameraPos;
RwV3d ClumpSpaceCameraAt;


/*
 *****************************************************************************
 */
void 
SkyDomeRotateSun(RwReal deltaZD, RwReal deltaAZ)
{
    SunZenithDist += deltaZD;

    /*
     * Limit the Sun's zenith distance to bewteen 0 and 85 degrees...
     */
    if( SunZenithDist > 85.0f )
    {
        SunZenithDist = 85.0f;
    }
    else if( SunZenithDist < 0.0f )
    {
        SunZenithDist = 0.0f;
    }

    SunAzimuth += deltaAZ;

    /*
     * Keep the Sun's azimuth in the range 0 --> 360 degrees...
     */
    if( SunAzimuth >= 360.0f )
    {
        SunAzimuth -= 360.0f;
    }
    else if( SunAzimuth < 0.0f )
    {
        SunAzimuth += 360.0f;
    }

    /*
     * What are the zenith values for this position of the Sun
     * and turbidity...
     */
    SkyDomeInitializeParams(SunZenithDist, SunAzimuth, Turbidity);

    /*
     * Determine the color of each vertex in the skydome...
     */
    SkyDomeCalculateColors(SunZenithDist, SunAzimuth, SkyDomeAtomic);

    return;
}


/*
 *****************************************************************************
 */
void 
UpdateClumpSpaceCameraParameters(void)
{
    RwMatrix *LTM = (RwMatrix *)NULL;
    RwMatrix *invLTM = (RwMatrix *)NULL;
    RwV3d *pos, *at;

    /*
     * Transform the camera's position and at-vector into clump-space
     * (fro wire-frame culling purposes)...
     */
    LTM = RwFrameGetLTM(RpClumpGetFrame(SkyDomeClump));
    invLTM = RwMatrixCreate();
    RwMatrixInvert(invLTM, LTM);

    pos = RwMatrixGetPos(RwFrameGetLTM(RwCameraGetFrame(Camera)));
    RwV3dTransformPoints(&ClumpSpaceCameraPos, pos, 1, invLTM);

    at = RwMatrixGetAt(RwFrameGetLTM(RwCameraGetFrame(Camera)));
    RwV3dTransformVectors(&ClumpSpaceCameraAt, at, 1, invLTM);

    RwMatrixDestroy(invLTM);

    return;
}


/*
 *****************************************************************************
 */
void 
SkyDomeRotate(RwReal xAngle, RwReal yAngle)
{
    RwMatrix *cameraMatrix;
    RwV3d right, up, pos;
    RwFrame *clumpFrame;

    cameraMatrix = RwFrameGetMatrix(RwCameraGetFrame(Camera));
    right = *RwMatrixGetRight(cameraMatrix);
    up = *RwMatrixGetUp(cameraMatrix);

    clumpFrame = RpClumpGetFrame(SkyDomeClump);
    pos = *RwMatrixGetPos(RwFrameGetMatrix(clumpFrame));

    /* 
     * First translate back to the origin...
     */
    RwV3dScale(&pos, &pos, -1.0f);
    RwFrameTranslate(clumpFrame, &pos, rwCOMBINEPOSTCONCAT);

    /*
     * Do the rotations...
     */
    RwFrameRotate(clumpFrame, &up, xAngle, rwCOMBINEPOSTCONCAT);
    RwFrameRotate(clumpFrame, &right, yAngle, rwCOMBINEPOSTCONCAT);

    /*
     * And translate back...
     */
    RwV3dScale(&pos, &pos, -1.0f);
    RwFrameTranslate(clumpFrame, &pos, rwCOMBINEPOSTCONCAT);

    /*
     * Update the clump-space position and at-vector of the camera...
     */
    UpdateClumpSpaceCameraParameters();

    return;
}


/*
 *****************************************************************************
 */
void 
SkyDomeTranslate(RwReal distance)
{
    RwMatrix *cameraMatrix;
    RwV3d at;
    RwFrame *clumpFrame;

    cameraMatrix = RwFrameGetMatrix(RwCameraGetFrame(Camera));
    at = *RwMatrixGetAt(cameraMatrix);

    clumpFrame = RpClumpGetFrame(SkyDomeClump);

    RwV3dScale(&at, &at, distance);
    RwFrameTranslate(clumpFrame, &at, rwCOMBINEPOSTCONCAT);

    /*
     * Update the clump-space position and at-vector of the camera...
     */
    UpdateClumpSpaceCameraParameters();

    return;
}


/*
 *****************************************************************************
 */
void 
CameraPoint(RwReal deltaElevation, RwReal deltaAzimuth)
{
    RwFrame *cameraFrame;
    RwV3d pos;
    RwV3d xAxis = {1.0f, 0.0f, 0.0f};
    RwV3d yAxis = {0.0f, 1.0f, 0.0f};

    CameraElevation += deltaElevation;

    /*
     * Limit the camera's elevation so that it never quite reaches
     * exactly +90 or -90 degrees to avoid the singularity in these
     * situations...
     */
    if( CameraElevation > 89.9f )
    {
        CameraElevation = 89.9f;
    }
    else if( CameraElevation < -89.9f )
    {
        CameraElevation = -89.9f;
    }

    CameraAzimuth += deltaAzimuth;

    /*
     * Keep the azimuth in the range 0 --> 360 degrees...
     */
    if( CameraAzimuth >= 360.0f )
    {
        CameraAzimuth -= 360.0f;
    }
    else if( CameraAzimuth < 0.0f )
    {
        CameraAzimuth += 360.0f;
    }

    cameraFrame = RwCameraGetFrame(Camera);

    /*
     * Remember where the camera is...
     */
    pos = *RwMatrixGetPos(RwFrameGetMatrix(cameraFrame));

    /*
     * Reset the camera's frame; we're dealing with absolute
     * orientations here...
     */
    RwFrameRotate(cameraFrame, &xAxis, 
        -CameraElevation, rwCOMBINEREPLACE);

    RwFrameRotate(cameraFrame, &yAxis, 
        CameraAzimuth, rwCOMBINEPOSTCONCAT);

    /*
     * Put the camera back to where it was...
     */
    RwFrameTranslate(cameraFrame, &pos, rwCOMBINEPOSTCONCAT);

    /*
     * Update the clump-space position and at-vector of the camera...
     */
    UpdateClumpSpaceCameraParameters();

    return;
}


/*
 *****************************************************************************
 */
void 
CameraUpdateHFoV(RwReal deltaHFoV)
{
    CameraHFoV += deltaHFoV;

    if( CameraHFoV < 1.0f )
    {
        CameraHFoV = 1.0f;
    }
    else if(CameraHFoV > 179.0f )
    {
        CameraHFoV = 179.0f;
    }

    CameraViewWindow = (RwReal)RwTan(CameraHFoV * rwPI / 360.0f);

    RsEventHandler(rsCAMERASIZE, (void *)NULL);

    return;
}

/*
 *****************************************************************************
 */
