
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 2000, 2001 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * skycolor.c
 *
 * Copyright (C) 2000, 2001 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: Sky dome creator for RW3.
 *
 ****************************************************************************/

#include <float.h>

#include "rwcore.h"
#include "rpworld.h"

#include "main.h"

#define MAX(x,y) (((x)>(y))?(x):(y))

typedef enum
{
    ColorXChromat = 0, 
    ColorYChromat, 
    ColorLuminance
}
ColorComponent;

typedef struct
{
    double X, Y, Z;
}
XYZColour;

static RwRGBA Black   = {  0,   0,   0, 255};
static RwRGBA SkyBlue = { 50, 100, 255, 255};
static RwRGBA White   = {255, 255, 255, 255};

static RwRGBA OutOfGamutColor = {0, 0, 0, 255};

static RwV3d Xaxis = {1.0f, 0.0f, 0.0f};
static RwV3d Yaxis = {0.0f, 1.0f, 0.0f};
static RwV3d Zaxis = {0.0f, 0.0f, 1.0f};

static double Perez[3][5];

static double ZenithXChromat;
static double ZenithYChromat;
static double ZenithLuminance;
static double XChromatZenithSkyFunctionValue;
static double YChromatZenithSkyFunctionValue;
static double LuminanceZenithSkyFunctionValue;

static XYZColour MonitorRGB[4] = 
{
    /* 
     * HP A1097C 
     */
    { 0.625, 0.340, 0.035 },    /* Red */
    { 0.285, 0.605, 0.110 },    /* Green */
    { 0.150, 0.065, 0.785 },    /* Blue */
    { 0.283, 0.298, 0.419 }     /* White point */
};

static double XYZtoRGB[3][3];
static double WhitePoint[3];

static RwRGBAReal *SkyColorReal = (RwRGBAReal *)NULL;


RwReal SunZenithDist = 0.0f;
RwReal SunAzimuth = 0.0f;
RwV3d SunPos;

RwReal Turbidity = 2.0f;
RwReal Exposure = 0.1f;


/*
 * NOTE ABOUT ZENITH DISTANCE, ELEVATION AND AZIMUTH (as used here).
 *
 * Zenith distance is measured from the Y-axis ('vertically'),
 *  increasing towards the X-Z plane.
 *
 * Elevation is measured from the X-Z plane ('vertically').
 *
 * Azimuth is measured from the Z-axis ('horizontally'), 
 *  increasing towards the X-axis.
 *
 * x = sin(ZD) * sin(AZ)  = cos(EL) * sin(AZ)
 *
 * y = cos(ZD)            = sin(EL)
 *
 * z = sin(ZD) * cos(AZ)  = cos(EL) * cos(AZ)
 *
 */



/*
 *****************************************************************************
 */
static RwBool 
MatrixInvert(double mat1[3][3], double mat2[3][3])
{
    double det;

    det = mat1[0][0] * mat1[1][1] * mat1[2][2] - mat1[0][0] * mat1[1][2] * mat1[2][1]
        + mat1[0][1] * mat1[1][2] * mat1[2][0] - mat1[0][1] * mat1[1][0] * mat1[2][2]
        + mat1[0][2] * mat1[1][0] * mat1[2][1] - mat1[0][2] * mat1[1][1] * mat1[2][0];

    if( det == 0.0 )
    {
        return FALSE;
    }

    mat2[0][0] =  (mat1[1][1] * mat1[2][2] - mat1[2][1] * mat1[1][2]) / det,
    mat2[0][1] = -(mat1[0][1] * mat1[2][2] - mat1[2][1] * mat1[0][2]) / det;
    mat2[0][2] =  (mat1[0][1] * mat1[1][2] - mat1[0][2] * mat1[1][1]) / det;
    mat2[1][0] = -(mat1[1][0] * mat1[2][2] - mat1[2][0] * mat1[1][2]) / det;
    mat2[1][1] =  (mat1[0][0] * mat1[2][2] - mat1[2][0] * mat1[0][2]) / det;
    mat2[1][2] = -(mat1[0][0] * mat1[1][2] - mat1[1][0] * mat1[0][2]) / det;
    mat2[2][0] =  (mat1[1][0] * mat1[2][1] - mat1[1][1] * mat1[2][0]) / det;
    mat2[2][1] = -(mat1[0][0] * mat1[2][1] - mat1[2][0] * mat1[0][1]) / det;
    mat2[2][2] =  (mat1[0][0] * mat1[1][1] - mat1[1][0] * mat1[0][1]) / det;

    return TRUE;
}



/*
 *****************************************************************************
 */
RwBool 
InitializeColor(void)
{	 
    int i, j;
    double c[3], chromats[3][3], invChromats[3][3];
    
    for(i=0; i<3; i++)
    {
        chromats[0][i] = MonitorRGB[i].X;
        chromats[1][i] = MonitorRGB[i].Y;
        chromats[2][i] = MonitorRGB[i].Z;
    }
	 
    if( !MatrixInvert(chromats, invChromats) )
    {
        return FALSE;
    }
    
    WhitePoint[0] = MonitorRGB[3].X / MonitorRGB[3].Y;
    WhitePoint[1] = 1.0;
    WhitePoint[2] = MonitorRGB[3].Z / MonitorRGB[3].Y;
	 
    for(i=0; i<3; i++)
    {
        c[i] = 0.0;

        for(j=0; j<3; j++)
        {
            c[i] += WhitePoint[j] * invChromats[i][j];
        }
    }
	 
    for(i=0; i<3; i++)
    {
        for(j=0; j<3; j++)
        {
            chromats[i][j] *= c[j];
        }
    }

    if( !MatrixInvert(chromats, XYZtoRGB) )
    {
        return FALSE;
    }

    return TRUE;
}



/*
 *****************************************************************************
 */
static void 
XYZ2RGB(RwRGBAReal *color, double X, double Y, double Z)
{
    color->red = 
        (RwReal)(XYZtoRGB[0][0] * X + XYZtoRGB[0][1] * Y + XYZtoRGB[0][2] * Z);
    
    color->green = 
        (RwReal)(XYZtoRGB[1][0] * X + XYZtoRGB[1][1] * Y + XYZtoRGB[1][2] * Z);
    
    color->blue = 
        (RwReal)(XYZtoRGB[2][0] * X + XYZtoRGB[2][1] * Y + XYZtoRGB[2][2] * Z);
    
    color->alpha = 0.0f;

    return;
}


/*
 *****************************************************************************
 */
void 
SkyDomeDestroySkyColorBuffer(void)
{
    if( (RwRGBAReal *)NULL != SkyColorReal )
    {
        RwFree(SkyColorReal);
    }

    return;
}


/*
 *****************************************************************************
 */
RwBool
SkyDomeResizeSkyColorBuffer(RwInt32 numVert)
{
    void *result;

    static RwInt32 vertexHWM = 0;

    /*
     * Grow the color buffer if required...
     */
    if( numVert > vertexHWM )
    {
        result = (void *)RwRealloc(SkyColorReal, numVert * sizeof(RwRGBAReal));

        if( (void *)NULL == result )
        {
            return FALSE;
        }
    }

    /*
     * If we get here we have enough buffer space...
     */
    if( numVert > vertexHWM )
    {
        SkyColorReal = (RwRGBAReal *)result;

        vertexHWM = numVert;
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static double 
SkyFunction(double theta, double gamma, ColorComponent comp)
{
    double result;
    double temp;

    temp = cos(gamma);
    temp = temp * temp;

    result = (1.0 + Perez[comp][0] * exp(Perez[comp][1] / cos(theta))) 
        * (1.0 + Perez[comp][2] * exp(Perez[comp][3] * gamma) + Perez[comp][4] * temp);

    if( !_finite(result) )
    {
        result = 0.0;
    }

    return result;
}


/*
 *****************************************************************************
 */
void 
SkyDomeInitializeParams(RwReal sunZD, RwReal sunAZ, RwReal turb)
{
    double sunZenithDist, sunAzimuth;
    double sunZenithDist2, sunZenithDist3;
    double turbidity, turbidity2;
    double chi;

    /*
     * Convert to radians (double valued)...
     */
    sunZenithDist = (double)sunZD * rwPI / 180.0;
    sunAzimuth = (double)sunAZ * rwPI / 180.0;

    /*
     * Calculate the 3D position of the Sun on the skydome...
     */
    SunPos.x = (RwReal)(RwSin(sunZenithDist) * RwSin(sunAzimuth));
    SunPos.y = (RwReal)RwCos(sunZenithDist);
    SunPos.z = (RwReal)(RwSin(sunZenithDist) * RwCos(sunAzimuth));

    turbidity = (double)turb;

    Perez[ColorXChromat][0] = -0.0193 * turbidity - 0.2592;
    Perez[ColorXChromat][1] = -0.0665 * turbidity + 0.0008;
    Perez[ColorXChromat][2] = -0.0004 * turbidity + 0.2125;
    Perez[ColorXChromat][3] = -0.0641 * turbidity - 0.8989;
    Perez[ColorXChromat][4] = -0.0033 * turbidity + 0.0452;

    Perez[ColorYChromat][0] = -0.0167 * turbidity - 0.2608;
    Perez[ColorYChromat][1] = -0.0950 * turbidity + 0.0092;
    Perez[ColorYChromat][2] = -0.0079 * turbidity + 0.2102;
    Perez[ColorYChromat][3] = -0.0441 * turbidity - 1.6537;
    Perez[ColorYChromat][4] = -0.0109 * turbidity + 0.0529;

    Perez[ColorLuminance][0] =  0.1787 * turbidity - 1.4630;
    Perez[ColorLuminance][1] = -0.3554 * turbidity + 0.4275;
    Perez[ColorLuminance][2] = -0.0227 * turbidity + 5.3251;
    Perez[ColorLuminance][3] =  0.1206 * turbidity - 2.5771;
    Perez[ColorLuminance][4] = -0.0670 * turbidity + 0.3703;

    sunZenithDist2 = sunZenithDist * sunZenithDist;
    sunZenithDist3 = sunZenithDist2 * sunZenithDist;

    turbidity2 = turbidity * turbidity;

    ZenithXChromat = 
        ( 0.00166 * sunZenithDist3 - 0.00375 * sunZenithDist2 + 0.00209 * sunZenithDist          ) * turbidity2 +
        (-0.02903 * sunZenithDist3 + 0.06377 * sunZenithDist2 - 0.03202 * sunZenithDist + 0.00394) * turbidity  +
        ( 0.11693 * sunZenithDist3 - 0.21196 * sunZenithDist2 + 0.06052 * sunZenithDist + 0.25886);

    ZenithYChromat = 
        ( 0.00275 * sunZenithDist3 - 0.00610 * sunZenithDist2 + 0.00317 * sunZenithDist          ) * turbidity2 +
        (-0.04214 * sunZenithDist3 + 0.08970 * sunZenithDist2 - 0.04153 * sunZenithDist + 0.00516) * turbidity  +
        ( 0.15346 * sunZenithDist3 - 0.26756 * sunZenithDist2 + 0.06670 * sunZenithDist + 0.26688);

    chi = ((4.0 / 9.0) - (turbidity / 120.0)) * (rwPI - (2.0 * sunZenithDist));

    ZenithLuminance =
        (((4.0453 * turbidity) - 4.9710) * tan(chi)) - (0.2155 * turbidity) + 2.4192;

    XChromatZenithSkyFunctionValue = 
        SkyFunction(0.0, sunZenithDist, ColorXChromat);

    YChromatZenithSkyFunctionValue = 
        SkyFunction(0.0, sunZenithDist, ColorYChromat);

    LuminanceZenithSkyFunctionValue = 
        SkyFunction(0.0, sunZenithDist, ColorLuminance);

    return;
}


/*
 *****************************************************************************
 */
static void 
GetViewDirection(RwV3d *direction, double *zenithDist, double *azimuth)
{
    RwReal dot;
    RwV3d projDir;

    /*
     * Get the zenith distance and azimuth from the normalized
     * direction vector.
     */

    /*
     * First the zenith distance...
     */
    dot = RwV3dDotProduct(direction, &Yaxis);

    /*
     * Make sure the dot product is not out of range (due to numerical
     * inccuracies) so the arc-cos doesn't crash out on us. Usually, if
     * it's outside the useable range, then it is just marginal...
     */
    if( dot > 1.0f )
    {
        dot = 1.0f;
    }
    else if( dot < -1.0f )
    {
        dot = -1.0f;
    }

    *zenithDist = RwACos(dot);

    /*
     * Now the azimuth. Project the direction onto the X-Z plane...
     */
    RwV3dScale(&projDir, &Yaxis, dot);
    RwV3dSub(&projDir, direction, &projDir);

    if( RwV3dLength(&projDir) > 0.0f )
    {
        RwV3dNormalize(&projDir, &projDir);
    }

    dot = RwV3dDotProduct(&projDir, &Zaxis);

    /*
     * Numerical inaccuracies here are more likely due the 
     * approximate square root use by RwV3dNormalize...
     */
    if( dot > 1.0f )
    {
        dot = 1.0f;
    }
    else if( dot < -1.0f )
    {
        dot = -1.0f;
    }

    *azimuth = RwACos(dot);

    /*
     * Adjust the azimuth to the range 0 --> 360 degrees...
     */
    if( RwV3dDotProduct(&projDir, &Xaxis) < 0.0f )
    {
        *azimuth = 2.0f * rwPI - (*azimuth);
    }

    return;
}



/*
 *****************************************************************************
 */
static RwRGBAReal 
GetSkyColor(double sunZenithDist, double sunAzimuth,
            double viewZenithDist, double viewAzimuth)
{
    double gamma;
    double xChromaticity, yChromaticity, luminance;
    double X, Y, Z;
    RwRGBAReal skyColor;

    gamma = viewAzimuth - sunAzimuth;
    gamma = cos(sunZenithDist) * cos(viewZenithDist) + 
        sin(sunZenithDist) * sin(viewZenithDist) * cos(gamma);

    gamma = RwACos(gamma);

    xChromaticity = ZenithXChromat * 
        SkyFunction(viewZenithDist, gamma, ColorXChromat) / 
        XChromatZenithSkyFunctionValue;

    yChromaticity = ZenithYChromat * 
        SkyFunction(viewZenithDist, gamma, ColorYChromat) / 
        YChromatZenithSkyFunctionValue;

    luminance = ZenithLuminance * 
        SkyFunction(viewZenithDist, gamma, ColorLuminance) / 
        LuminanceZenithSkyFunctionValue;

    X = xChromaticity * luminance / yChromaticity;

    Y = luminance;

    Z = luminance / yChromaticity - X - Y;

    XYZ2RGB(&skyColor, X, Y, Z);

    return skyColor;
}


/*
 *****************************************************************************
 */
static void 
ExposeFunction(RwRGBAReal *light, RwRGBA *color)
{
    color->red = 
        (RwUInt8)(255.0f * (1.0f - (RwReal)RwExp(light->red * -Exposure)));
    
    color->green = 
        (RwUInt8)(255.0f * (1.0f - (RwReal)RwExp(light->green * -Exposure)));
    
    color->blue = 
        (RwUInt8)(255.0f * (1.0f - (RwReal)RwExp(light->blue * -Exposure)));
    
    color->alpha = 255;

    return;
}


/*
 *****************************************************************************
 */
void 
SkyDomeSetPrelights(RpAtomic *atomic, RwBool skyColorsOn)
{
    RpGeometry *geometry;
    RwInt32 numVert, i;
    RwRGBA *prelight;

    geometry = RpAtomicGetGeometry(atomic);

    numVert = RpGeometryGetNumVertices(geometry);
    prelight = RpGeometryGetPreLightColors(geometry);

    RpGeometryLock(geometry, rpGEOMETRYLOCKPRELIGHT);

    if( skyColorsOn )
    {
        for(i=0; i<numVert; i++)
        {
            if( SkyColorReal[i].red < 0.0f   || 
                SkyColorReal[i].green < 0.0f || 
                SkyColorReal[i].blue < 0.0f )
            {
                prelight[i] = OutOfGamutColor;
            }
            else
            {
                ExposeFunction(&SkyColorReal[i], &prelight[i]);
            }
        }
    }
    else
    {
        for(i=0; i<numVert; i++)
        {
            prelight[i] = Black;
        }
    }

    RpGeometryUnlock(geometry);

    return;
}



/*
 *****************************************************************************
 */
void 
SkyDomeCalculateColors(RwReal sunZD, RwReal sunAZ, RpAtomic *atomic)
{
    RwInt32 i;
    RpGeometry *geometry;
    RpMorphTarget *morphTarget;
    RwV3d *vlist;
    RwInt32 numVert;
    double sunZenithDist, sunAzimuth;

    /*
     * Convert to radians (double valued)...
     */
    sunZenithDist = (double)sunZD * rwPI / 180.0;
    sunAzimuth = (double)sunAZ * rwPI / 180.0;

    geometry = RpAtomicGetGeometry(atomic);
    morphTarget = RpGeometryGetMorphTarget(geometry, 0);

    numVert = RpGeometryGetNumVertices(geometry);
    vlist = RpMorphTargetGetVertices(morphTarget);

    for(i=0; i<numVert; i++)
    {
        double viewZenithDist, viewAzimuth;

        GetViewDirection(vlist, &viewZenithDist, &viewAzimuth);

        SkyColorReal[i] = 
            GetSkyColor(sunZenithDist, sunAzimuth, viewZenithDist, viewAzimuth);

        vlist++;
    }

    /*
     * Bring the sky colors into the monitor's gamut and use 
     * them to set the prelights on the skydome clump...
     */
    SkyDomeSetPrelights(atomic, SkyColorsOn);

    return;
}

/*
 *****************************************************************************
 */

