
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 2000, 2001 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * render.c
 *
 * Copyright (C) 2000, 2001 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: Sky dome creator for RW3.
 *
 ****************************************************************************/

#include "rwcore.h"
#include "rpworld.h"

#ifdef RWLOGO
#include "rplogo.h"
#endif

#include "rtcharse.h"

#include "skeleton.h"
#include "menu.h"

#ifdef RWMETRICS
#include "metrics.h"
#endif

#include "flare.h"
#include "main.h"
#include "skycolor.h"
#include "transform.h"
#include "events.h"

static RwIm3DVertex *Im3DVertices = (RwIm3DVertex *)NULL;
static RwImVertexIndex *Im3DLineIndices = (RwImVertexIndex *)NULL;

static RwV3d SquareVertices[4] = 
{
    {  1.1f,  0.0f, -1.1f },
    { -1.1f,  0.0f, -1.1f },
    { -1.1f,  0.0f,  1.1f },
    {  1.1f,  0.0f,  1.1f }
};

static RwTexCoords SquareUVs[4] = 
{
    { 0.0f, 0.0f },
    { 1.0f, 0.0f },
    { 1.0f, 1.0f },
    { 0.0f, 1.0f }
};

static RwRGBA WireFrameColor = {160,  64, 0, 255};
static RwRGBA NormalsColor = { 64, 160, 0, 255};
static RwRGBA GroundColor = {7, 77, 48, 255};
static RwRGBA White = {255, 255, 255, 255};

static RwReal NormalsScaleFactor = 0.2f;


/*
 *****************************************************************************
 */
void 
DestroyIm3DBuffers(void)
{
    if( (RwIm3DVertex *)NULL != Im3DVertices )
    {
        RwFree(Im3DVertices);
    }

    if( (RwImVertexIndex *)NULL != Im3DLineIndices )
    {
        RwFree(Im3DLineIndices);
    }

    return;
}


/*
 *****************************************************************************
 */
static RwBool
ResizeIm3DBuffers(RwInt32 numVertices, RwInt32 numIndices)
{
    void *vertices;
    void *indices;

    static RwInt32 vertexHWM = 0;
    static RwInt32 indexHWM = 0;

    /*
     * Grow the vertex buffer if required...
     */
    if( numVertices > vertexHWM )
    {
        vertices = RwRealloc(Im3DVertices, numVertices * sizeof(RwIm3DVertex));

        if( (void *)NULL == vertices )
        {
            return FALSE;
        }
    }

    /*
     * Grow the index buffer if required...
     */
    if( numIndices > indexHWM )
    {
        indices = RwRealloc(Im3DLineIndices, numIndices * sizeof(RwImVertexIndex));

        if( (void *)NULL == indices )
        {
            RwFree(vertices);

            return FALSE;
        }
    }

    /*
     * If we get here we have enough buffer space to 
     * performed the required IMM rendering...
     */
    if( numVertices > vertexHWM )
    {
        Im3DVertices = (RwIm3DVertex *)vertices;

        vertexHWM = numVertices;
    }

    if( numIndices > indexHWM )
    {
        Im3DLineIndices = (RwImVertexIndex *)indices;

        indexHWM = numIndices;
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static void 
RenderUpperGroundPlane(void)
{
    RwMatrix *LTM;
    RwRGBA color;
    RwV3d *pos;
    RwInt32 vert;

    /*
     * Make sure we have sufficient buffer space...
     */
    if( !ResizeIm3DBuffers(6, 0) )
    {
        return;
    }

    color = White;

    vert = 0;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[0], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[0], 
        color.red, color.green, color.blue, color.alpha);
    RwIm3DVertexSetU(&Im3DVertices[0], SquareUVs[vert].u);
    RwIm3DVertexSetV(&Im3DVertices[0], SquareUVs[vert].v);

    vert = 1;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[1], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[1], 
        color.red, color.green, color.blue, color.alpha);
    RwIm3DVertexSetU(&Im3DVertices[1], SquareUVs[vert].u);
    RwIm3DVertexSetV(&Im3DVertices[1], SquareUVs[vert].v);

    vert = 2;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[2], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[2], 
        color.red, color.green, color.blue, color.alpha);
    RwIm3DVertexSetU(&Im3DVertices[2], SquareUVs[vert].u);
    RwIm3DVertexSetV(&Im3DVertices[2], SquareUVs[vert].v);


    vert = 0;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[3], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[3], 
        color.red, color.green, color.blue, color.alpha);
    RwIm3DVertexSetU(&Im3DVertices[3], SquareUVs[vert].u);
    RwIm3DVertexSetV(&Im3DVertices[3], SquareUVs[vert].v);

    vert = 2;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[4], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[4], 
        color.red, color.green, color.blue, color.alpha);
    RwIm3DVertexSetU(&Im3DVertices[4], SquareUVs[vert].u);
    RwIm3DVertexSetV(&Im3DVertices[4], SquareUVs[vert].v);

    vert = 3;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[5], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[5], 
        color.red, color.green, color.blue, color.alpha);
    RwIm3DVertexSetU(&Im3DVertices[5], SquareUVs[vert].u);
    RwIm3DVertexSetV(&Im3DVertices[5], SquareUVs[vert].v);

    LTM = RwFrameGetLTM(RpClumpGetFrame(SkyDomeClump));

    if( RwIm3DTransform(Im3DVertices, 6, LTM, rwIM3D_VERTEXUV | rwIM3D_ALLOPAQUE) != (void *)NULL)
    {
        RwIm3DRenderPrimitive(rwPRIMTYPETRILIST);

        RwIm3DEnd();
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
RenderLowerGroundPlane(void)
{
    RwMatrix *LTM;
    RwRGBA color;
    RwV3d *pos;
    RwInt32 vert;

    /*
     * Make sure we have sufficient buffer space...
     */
    if( !ResizeIm3DBuffers(6, 0) )
    {
        return;
    }

    color = WireFrameColor;

    vert = 0;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[0], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[0], 
        color.red, color.green, color.blue, color.alpha);

    vert = 2;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[1], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[1], 
        color.red, color.green, color.blue, color.alpha);

    vert = 1;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[2], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[2], 
        color.red, color.green, color.blue, color.alpha);


    vert = 0;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[3], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[3], 
        color.red, color.green, color.blue, color.alpha);

    vert = 3;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[4], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[4], 
        color.red, color.green, color.blue, color.alpha);

    vert = 2;
    pos = &SquareVertices[vert];
    RwIm3DVertexSetPos(&Im3DVertices[5], pos->x, pos->y, pos->z);
    RwIm3DVertexSetRGBA(&Im3DVertices[5], 
        color.red, color.green, color.blue, color.alpha);

    LTM = RwFrameGetLTM(RpClumpGetFrame(SkyDomeClump));

    if( RwIm3DTransform(Im3DVertices, 6, LTM, rwIM3D_ALLOPAQUE) != (void *)NULL )
    {
        RwIm3DRenderPrimitive(rwPRIMTYPETRILIST);

        RwIm3DEnd();
    }

    return;
}


/*
 *****************************************************************************
 */
static RwBool 
CameraFacingVertexNormal(RwV3d *normal, RwV3d *pos)
{
    RwV3d v;

    RwV3dSub(&v, pos, &ClumpSpaceCameraPos);

    if( RwV3dDotProduct(&v, &ClumpSpaceCameraAt) < 0.0f )
    {
        /*
         * Vertex behind the camera...
         */
        return FALSE;
    }
    else
    {
        if( RwV3dDotProduct(&v, normal) >= 0.0f )
        {
            /*
             * Normal points away from the camera...
             */
            return FALSE;
        }
        else
        {
            /*
             * Normal points towards the camera...
             */
            return TRUE;
        }
    }
}


/*
 *****************************************************************************
 */
static RwBool 
CameraFacingLineSegment(RwV3d *v0, RwV3d *v1, RwV3d *n0, RwV3d *n1)
{
    RwV3d pos, normal;

    /*
     * Calculate the mid-point and average of the normals...
     */
    RwV3dAdd(&pos, v0, v1);
    RwV3dScale(&pos, &pos, 0.5f);

    RwV3dAdd(&normal, n0, n1);
    RwV3dScale(&normal, &normal, 0.5f);

    return CameraFacingVertexNormal(&normal, &pos);
}


/*
 *****************************************************************************
 */
static void 
RenderWireMesh(void)
{
    RpGeometry *geometry;
    RpMorphTarget *morphTarget;
    RwV3d *vlist, *nlist;
    RwMatrix *LTM;
    RwInt32 i, j, k;
    RwRGBA color;
    RwInt32 numVertices, numIndices;
    RwInt32 poleV0, poleV1;

    numVertices = AzRes * ElRes + 1;
    numIndices = 4 * ElRes * AzRes;

    /*
     * Make sure we have sufficient buffer space.
     * As we're culling back-facing immediate mode line segments, we do not
     * know in advance how many are drawn as this depends on the view.
     * So we allocate enough space to draw them all...
     */
    if( !ResizeIm3DBuffers(numVertices, numIndices) )
    {
        return;
    }

    geometry = RpAtomicGetGeometry(SkyDomeAtomic);
    morphTarget = RpGeometryGetMorphTarget(geometry, 0);
    vlist = RpMorphTargetGetVertices(morphTarget);
    nlist = RpMorphTargetGetVertexNormals(morphTarget);

    color = WireFrameColor;

    /*
     * Setup all vertices (even though not all, if any, will
     * be referenced after line segment culling)...
     */
    for(k=0; k<numVertices; k++)
    {
        RwIm3DVertexSetRGBA(&Im3DVertices[k], 
            color.red, color.green, color.blue, color.alpha);

        RwIm3DVertexSetPos(&Im3DVertices[k], vlist[k].x, vlist[k].y, vlist[k].z);
    }

    /*
     * Setup the indices for the lines of constant elevation...
     */
    k = 0;
    for(j=0; j<ElRes; j++)
    {
        RwInt32 linkV0, linkV1;

        for(i=0; i<AzRes-1; i++)
        {
            RwInt32 v0, v1;

            v0 = i + j * AzRes;
            v1 = v0 + 1;

            if( 0 == i )
            {
                linkV1 = v0;
            }
            else if( AzRes - 2 == i )
            {
                linkV0 = v1;
            }

            if( CameraFacingLineSegment(&vlist[v0], &vlist[v1], &nlist[v0], &nlist[v1]) )
            {
                Im3DLineIndices[k] = (RwImVertexIndex)v0;
                Im3DLineIndices[k + 1] = (RwImVertexIndex)v1;

                k += 2;
            }
        }

        if( CameraFacingLineSegment(&vlist[linkV0], &vlist[linkV1], &nlist[linkV0], &nlist[linkV1]) )
        {
            Im3DLineIndices[k] = (RwImVertexIndex)linkV0;
            Im3DLineIndices[k + 1] = (RwImVertexIndex)linkV1;

            k += 2;
        }
    }

    /*
     * Setup the indices for the lines of constant azimuth...
     */
    for(j=0; j<AzRes; j++)
    {
        for(i=0; i<ElRes-1; i++)
        {
            RwInt32 v0, v1;

            v0 = j + i * AzRes;
            v1 = v0 + AzRes;

            if( CameraFacingLineSegment(&vlist[v0], &vlist[v1], &nlist[v0], &nlist[v1]) )
            {
                Im3DLineIndices[k] = (RwImVertexIndex)v0;
                Im3DLineIndices[k + 1] = (RwImVertexIndex)v1;

                k += 2;
            }
        }
    }

    /*
     * Setup the indices for the azimuth lines around the pole...
     */
    poleV0 = AzRes * (ElRes - 1);
    poleV1 = numVertices - 1;
    for(i=0; i<AzRes; i++)
    {
        if( CameraFacingLineSegment(&vlist[poleV0], &vlist[poleV1], &nlist[poleV0], &nlist[poleV1]) )
        {
            Im3DLineIndices[k] = (RwImVertexIndex)poleV0;
            Im3DLineIndices[k + 1] = (RwImVertexIndex)poleV1;

            k += 2;
        }

        poleV0++;
    }

    numIndices = k;

    /* 
     * Guard against there being no lines segments to render, even though
     * we've a bunch of vertices to transform...
     */
    if( numIndices < 2 )
    {
        return;
    }

    /*
     * Transform and render...
     */
    LTM = RwFrameGetLTM(RpClumpGetFrame(SkyDomeClump));

    if( RwIm3DTransform(Im3DVertices, numVertices, LTM, rwIM3D_ALLOPAQUE) )
    {
        RwIm3DRenderIndexedPrimitive(rwPRIMTYPELINELIST, 
            Im3DLineIndices, numIndices);

        RwIm3DEnd();
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
RenderVertexNormals(void)
{
    RpGeometry *geometry;
    RpMorphTarget *morphTarget;
    RwIm3DVertex *imVertex;
    RwInt32 numVertices, numImmVert;
    RwV3d *vertPos;
    RwV3d *vertNormal;
    RwMatrix *LTM;
    RwInt32 i;
    RwRGBA color;

    geometry = RpAtomicGetGeometry(SkyDomeAtomic);
    numVertices = RpGeometryGetNumVertices(geometry);

    /*
     * Make sure we have sufficient buffer space.
     * As we're culling back-facing immediate mode normals, we do not
     * know in advance how many are drawn as this depends on the view.
     * So we allocate enough space to draw them all...
     */
    if( !ResizeIm3DBuffers(2 * numVertices, 0) )
    {
        return;
    }

    morphTarget = RpGeometryGetMorphTarget(geometry, 0);
    vertPos = RpMorphTargetGetVertices(morphTarget);
    vertNormal = RpMorphTargetGetVertexNormals(morphTarget);

    color = NormalsColor;

    numImmVert = 0;
    imVertex = Im3DVertices;

    for(i=0; i<numVertices; i++)
    {
        RwV3d pos, normal;

        pos = vertPos[i];
        normal = vertNormal[i];

        if( CameraFacingVertexNormal(&normal, &pos) )
        {
            RwIm3DVertexSetPos(imVertex, pos.x, pos.y, pos.z);

            RwIm3DVertexSetRGBA(imVertex, 
                color.red, color.green, color.blue, color.alpha);

            imVertex++;

            RwIm3DVertexSetPos(imVertex,
                pos.x + NormalsScaleFactor * normal.x, 
                pos.y + NormalsScaleFactor * normal.y, 
                pos.z + NormalsScaleFactor * normal.z
            );

            RwIm3DVertexSetRGBA(imVertex, 
                color.red, color.green, color.blue, color.alpha);

            imVertex++;

            numImmVert += 2;
        }
    }

    if( numImmVert < 2 )
    {
        return;
    }

    /*
     * Transform and render...
     */
    LTM = RwFrameGetLTM(RpClumpGetFrame(SkyDomeClump));

    if( RwIm3DTransform(Im3DVertices, numImmVert, LTM, rwIM3D_ALLOPAQUE) != (void *)NULL )
    {
        RwIm3DRenderPrimitive(rwPRIMTYPELINELIST);

        RwIm3DEnd();
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
RenderDisplayData(void)
{
    RwChar caption[256];
    RtCharsetDesc charsetDesc;
    RwRaster *camRas;
    RwInt32 crw, crh;
    RpGeometry *geometry;

    RtCharsetGetDesc(Charset, &charsetDesc);

    camRas = RwCameraGetRaster(Camera);
    crw = RwRasterGetWidth(camRas);
    crh = RwRasterGetHeight(camRas);

    /*
     * Print the Sun's zenith distance and azimuth...
     */
    rwsprintf(caption, RWSTRING("Sun ZD: %.1f"), SunZenithDist);
    RtCharsetPrint(Charset, caption,
        crw - 130, charsetDesc.height);

    rwsprintf(caption, RWSTRING("Sun AZ: %.1f"), SunAzimuth);
    RtCharsetPrint(Charset, caption,
        crw - 130, 2 * charsetDesc.height);

    if( PreviewOn )
    {
        /*
         * Print the camera's elevation and azimuth...
         */
        rwsprintf(caption, RWSTRING("Cam EL: %.1f"), CameraElevation);
        RtCharsetPrint(Charset, caption,
            crw - 130, 4 * charsetDesc.height);

        rwsprintf(caption, RWSTRING("Cam AZ: %.1f"), CameraAzimuth);
        RtCharsetPrint(Charset, caption,
            crw - 130, 5 * charsetDesc.height);

        rwsprintf(caption, RWSTRING("Cam HF: %.1f"), CameraHFoV);
        RtCharsetPrint(Charset, caption,
            crw - 130, 6 * charsetDesc.height);
    }

    /*
     * Print the number of vertices and triangles...
     */
    geometry = RpAtomicGetGeometry(SkyDomeAtomic);

    rwsprintf(caption, RWSTRING(" Vertices: %3d"), 
        RpGeometryGetNumVertices(geometry));

    RtCharsetPrint(Charset, caption, 
        crw - 135, crh - 2 * charsetDesc.height - 20); 

    rwsprintf(caption, RWSTRING("Triangles: %3d"), 
        RpGeometryGetNumTriangles(geometry));

    RtCharsetPrint(Charset, caption, 
        crw - 135, crh - charsetDesc.height - 20);

    /*
     * Print the FPS if visible...
     */
    if( FPSOn )
    {
        rwsprintf(caption, RWSTRING("FPS: %03d"), FramesPerSecond);

        RtCharsetPrint(Charset, caption, 
            (crw - charsetDesc.width * rwstrlen(caption)) / 2,
            charsetDesc.height);
    }

    return;
}


/*
 *****************************************************************************
 */
void 
Render(void)
{
    RwRGBA *clearColor;

    if( PreviewOn )
    {
        clearColor = &GroundColor;
    }
    else
    {
        clearColor = &BackgroundColor;
    }

    RwCameraClear(Camera, clearColor, rwCAMERACLEARZ|rwCAMERACLEARIMAGE);

    if( RwCameraBeginUpdate(Camera) != (RwCamera *)NULL )
    {
        if( RdMenuGetStatus() != rdHELPMODE )
        {
            RwRenderStateSet(rwRENDERSTATESHADEMODE, (void *)rwSHADEMODEGOURAUD);
            RwRenderStateSet(rwRENDERSTATEZTESTENABLE, (void *)TRUE);
            RwRenderStateSet(rwRENDERSTATEZWRITEENABLE, (void *)TRUE);

            RpWorldRender(World);

            if( !PreviewOn && GroundOn )
            {
                RwRenderStateSet(rwRENDERSTATETEXTURERASTER, 
                                    (void *)RwTextureGetRaster(GroundTex));
                
                RenderUpperGroundPlane();

                RwRenderStateSet(rwRENDERSTATETEXTURERASTER, (void *)NULL);
                RwRenderStateSet(rwRENDERSTATESHADEMODE, (void *)rwSHADEMODEFLAT);

                RenderLowerGroundPlane();
            }

            if( WireMeshOn || NormalsOn )
            {
                RwRenderStateSet(rwRENDERSTATETEXTURERASTER, (void *)NULL);
                RwRenderStateSet(rwRENDERSTATESHADEMODE, (void *)rwSHADEMODEFLAT);
                RwRenderStateSet(rwRENDERSTATEZTESTENABLE, (void *)FALSE);
                RwRenderStateSet(rwRENDERSTATEZWRITEENABLE, (void *)FALSE);

                if( WireMeshOn )
                {
                    RenderWireMesh();
                }
                
                if( !PreviewOn && NormalsOn )
                {
                    RenderVertexNormals();
                }
            }

            if( PreviewOn && LensEffectOn && SkyColorsOn )
            {
                if( UpdateFlares() )
                {
                    RwRenderStateSet(rwRENDERSTATEZTESTENABLE, (void *)FALSE);
                    RwRenderStateSet(rwRENDERSTATEZWRITEENABLE, (void *)FALSE);
                    RwRenderStateSet(rwRENDERSTATESRCBLEND, (void *)rwBLENDONE);
                    RwRenderStateSet(rwRENDERSTATEDESTBLEND, (void *)rwBLENDONE);

                    RenderFlares();
                }
            }

            if( DisplayDataOn )
            {
                RenderDisplayData();
            }
        }

        RdMenuRender(Camera, (RtCharset *)NULL);

#ifdef RWMETRICS
        RsMetricsRender();
#endif

        RwCameraEndUpdate(Camera);
    }

    if( (RdMenuGetStatus() != rdHELPMODE) && CameraPointing )
    {
        RwRaster *camRas;

        camRas = RwCameraGetRaster(Camera);

        RwRasterPushContext(camRas);

        RwRasterRender(Pointer, 
            (RwRasterGetWidth(camRas) - RwRasterGetWidth(Pointer)) >> 1, 
            (RwRasterGetHeight(camRas) - RwRasterGetHeight(Pointer)) >> 1);

        RwRasterPopContext();
    }

    /* 
     * Display camera's raster...
     */
    RsCameraShowRaster(Camera);

    FrameCounter++;

    return;
}

/*
 *****************************************************************************
 */
