
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 2000, 2001 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * main.c
 *
 * Copyright (C) 2000, 2001 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: Sky dome creator for RW3.
 *
 ****************************************************************************/

#include "rwcore.h"
#include "rpworld.h"

#ifdef RWLOGO
#include "rplogo.h"
#endif

#include "rtcharse.h"

#include "skeleton.h"
#include "menu.h"
#include "camera.h"

#ifdef RWMETRICS
#include "metrics.h"
#endif

#include "flare.h"
#include "geometry.h"
#include "render.h"
#include "skycolor.h"
#include "transform.h"
#include "events.h"

#define DEFAULT_SCREEN_WIDTH (640)
#define DEFAULT_SCREEN_HEIGHT (480)

#define DEFAULT_ASPECTRATIO (4.0f/3.0f)

#define INITHFOV (45.0f)

#define SPINRATE (15.0f)


static RwMatrix CameraOutsideLTM;
static RwMatrix CameraInsideLTM;
static RwMatrix SkyDomeOutsideLTM;
static RwMatrix SkyDomeInsideLTM;
static RwReal CameraInsideHFoV = 90.0f;

static RwBool CameraSpinOn = FALSE;

static RwInt32 TessellationRes = 9;


RwRaster *Pointer = (RwRaster *)NULL;

RwBool FPSOn = FALSE;

RwInt32 FrameCounter = 0;
RwInt32 FramesPerSecond = 0;

RwRGBA ForegroundColor = {200, 200, 200, 255};
RwRGBA BackgroundColor = { 64,  64,  64,   0};

RwBool SkyColorsOn = TRUE;
RwBool GroundOn = TRUE;
RwBool WireMeshOn = FALSE;
RwBool NormalsOn = FALSE;
RwBool PreviewOn = FALSE;
RwBool LensEffectOn = TRUE;
RwBool DisplayDataOn = TRUE;

RpWorld *World = (RpWorld *)NULL;
RwCamera *Camera = (RwCamera *)NULL;
RtCharset *Charset = (RtCharset *)NULL;
RwTexture *GroundTex = (RwTexture *)NULL;

RpClump *SkyDomeClump = (RpClump *)NULL;
RpAtomic *SkyDomeAtomic = (RpAtomic *)NULL;

RwReal CameraViewWindow = 1.0f;
RwReal CameraHFoV = INITHFOV;

RwInt32 AzRes;
RwInt32 ElRes;



/*
 *****************************************************************************
 */
static RpWorld *
CreateWorld(void)
{
    RpWorld *world;
    RwBBox bb;

    bb.inf.x = bb.inf.y = bb.inf.z = -100.0f;
    bb.sup.x = bb.sup.y = bb.sup.z = 100.0f;

    world = RpWorldCreate(&bb);

    return world;
}


/*
 *****************************************************************************
 */
static RwCamera *
CreateCamera(RpWorld *world)
{
    RwCamera *camera;

    camera = RdCameraCreate(RsGlobal.maximumWidth, RsGlobal.maximumHeight, TRUE);
    if( (RwCamera *)NULL != camera )
    {
        RwCameraSetNearClipPlane(camera, 0.1f);
        RwCameraSetFarClipPlane(camera, 30.0f);

        RpWorldAddCamera(world, camera);

        return camera;
    }

    return (RwCamera *)NULL;
}


/*
 *****************************************************************************
 */
static RpClump *
CreateClump(RpWorld *world)
{
    RpClump *clump = (RpClump *)NULL;

    clump = RpClumpCreate();
    if( (RpClump *)NULL != clump )
    {
        RwFrame *clumpFrame = (RwFrame *)NULL;

        clumpFrame = RwFrameCreate();
        if( (RwFrame *)NULL != clumpFrame )
        {
            RpAtomic *atomic = (RpAtomic *)NULL;

            RpClumpSetFrame(clump, clumpFrame);

            atomic = RpAtomicCreate();
            if( (RpAtomic *)NULL != atomic )
            {
                RwFrame *atomicFrame = (RwFrame *)NULL;

                atomicFrame = RwFrameCreate();
                if( (RwFrame *)NULL != atomicFrame )
                {
                    RpGeometry *geometry = (RpGeometry *)NULL;

                    RpAtomicSetFrame(atomic, atomicFrame);

                    geometry = SkyDomeCreateGeometry(TessellationRes);
                    if( (RpGeometry *)NULL != geometry )
                    {
                        RwInt32 numVertices;

                        RpAtomicSetGeometry(atomic, geometry, 0);
                        RpGeometryDestroy(geometry);

                        numVertices = RpGeometryGetNumVertices(geometry);

                        if( SkyDomeResizeSkyColorBuffer(numVertices) )
                        {
                            RwFrameAddChild(clumpFrame, atomicFrame);
                            RpClumpAddAtomic(clump, atomic);
                            RpWorldAddClump(world, clump);

                            SkyDomeAtomic = atomic;

                            return clump;
                        }
                    }

                    RpAtomicSetFrame(atomic, (RwFrame *)NULL);
                    RwFrameDestroy(atomicFrame);
                }

                RpAtomicDestroy(atomic);
            }

            RpClumpSetFrame(clump, (RwFrame *)NULL);
            RwFrameDestroy(clumpFrame);
        }

        RpClumpDestroy(clump);
    }

    return (RpClump *)NULL;
}


/*
 *****************************************************************************
 */
static RwTexture * 
CreateGroundTexture(void)
{
    RwChar *path;
    RwTexture *texture;

    path = RsPathnameCreate(RWSTRING("textures/"));
    RwImageSetPath(path);
    RsPathnameDestroy(path);

    texture = RwTextureRead(RWSTRING("ground"), (RwChar *)NULL);
    if( (RwTexture *)NULL != texture )
    {
        RwTextureSetFilterMode(texture, rwFILTERLINEAR);

        return texture;
    }

    return (RwTexture *)NULL;
}


/*
 *****************************************************************************
 */
static RwBool 
Initialize(void)
{
    if( RsInitialize() )
    {
        if( 0 == RsGlobal.maximumWidth )
        {
            RsGlobal.maximumWidth = DEFAULT_SCREEN_WIDTH;
        }

        if( 0 == RsGlobal.maximumHeight )
        {
            RsGlobal.maximumHeight = DEFAULT_SCREEN_HEIGHT;
        }

        RsGlobal.appName = RWSTRING("RW3 Skydome Creator");

        RsGlobal.maxFPS = 100;

        return TRUE;
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static RwBool 
TessellateCallback(RwBool testEnable)
{
    RpGeometry *geometry;

    if( testEnable )
    {
        return TRUE;
    }

    /* 
     * Attempt to create a skydome geometry at the requested resolution...
     */
    geometry = SkyDomeCreateGeometry(TessellationRes);
    if( (RpGeometry *)NULL != geometry )
    {
        /*
         * We're got the geometry, is there enough space for the 
         * sky color buffer...
         */
        if( SkyDomeResizeSkyColorBuffer(RpGeometryGetNumVertices(geometry)) )
        {
            /*
             * This will destroy the old geometry...
             */
            RpAtomicSetGeometry(SkyDomeAtomic, geometry, 0);
            
            /*
             * Remove the application's ownership of the geometry now that
             * the atomic has it...
             */
            RpGeometryDestroy(geometry);

            /*
             * Determine the color of each vertex in the skydome...
             */
            SkyDomeCalculateColors(SunZenithDist, SunAzimuth, SkyDomeAtomic);

            if( PreviewOn )
            {
                SkyDomeInvertGeometry();
            }

            ElRes = TessellationRes;
            AzRes = ElRes * 4;
        }
        else
        {
            /*
             * Not enough sky color buffer space, so stick with what we have...
             */
            RpGeometryDestroy(geometry);

            RsErrorMessage(RWSTRING("Tessellation failed."));
        }
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
CurrentExposureCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return SkyColorsOn;
    }

    SkyDomeSetPrelights(SkyDomeAtomic, TRUE);

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
PreviewToggleCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return TRUE;
    }

    SkyDomeInvertGeometry();

    if( PreviewOn )
    {
        /*
         * Save the outside positions and orientations...
         */
        RwMatrixCopy(&CameraOutsideLTM, RwFrameGetLTM(RwCameraGetFrame(Camera)));
        RwMatrixCopy(&SkyDomeOutsideLTM, RwFrameGetLTM(RpClumpGetFrame(SkyDomeClump)));

        /*
         * Set the inside positions and orientations...
         */
        RwFrameTransform(RwCameraGetFrame(Camera), 
            &CameraInsideLTM, rwCOMBINEREPLACE);
        
        RwFrameTransform(RpClumpGetFrame(SkyDomeClump), 
            &SkyDomeInsideLTM, rwCOMBINEREPLACE);

        /*
         * Set the inside HFoV...
         */
        CameraHFoV = CameraInsideHFoV;
        CameraUpdateHFoV(0.0f);
    }
    else
    {
        /*
         * Save the inside positions and orientations...
         */
        RwMatrixCopy(&CameraInsideLTM, RwFrameGetLTM(RwCameraGetFrame(Camera)));
        RwMatrixCopy(&SkyDomeInsideLTM, RwFrameGetLTM(RpClumpGetFrame(SkyDomeClump)));
        
        CameraInsideHFoV = CameraHFoV;

        /*
         * Set the outside positions and orientations...
         */
        RwFrameTransform(RwCameraGetFrame(Camera), 
            &CameraOutsideLTM, rwCOMBINEREPLACE);
        
        RwFrameTransform(RpClumpGetFrame(SkyDomeClump), 
            &SkyDomeOutsideLTM, rwCOMBINEREPLACE);

        /*
         * Set the outside HFoV...
         */
        CameraHFoV = INITHFOV;
        CameraUpdateHFoV(0.0f);
    }

    /*
     * Update the skydome-space position and at-vector of the camera...
     */
    UpdateClumpSpaceCameraParameters();

    CameraSpinOn = FALSE;

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
SkyColorsToggleCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return TRUE;
    }

    SkyDomeSetPrelights(SkyDomeAtomic, SkyColorsOn);

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
GroundToggleCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return !PreviewOn;
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
NormalsToggleCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return !PreviewOn;
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
LensEffectToggleCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return PreviewOn && SkyColorsOn;
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
CameraSpinToggleCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return PreviewOn;
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
SkyDomeSaveCallback(RwBool testEnable)
{
    RwChar *path;
    RwStream *stream;

    if( testEnable )
    {
        return SkyColorsOn;
    }

    path = RsPathnameCreate(RWSTRING("skydome.dff"));
    stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMWRITE, path);
    RsPathnameDestroy(path);

    if( (RwStream *)NULL != stream )
    {
        RpClumpStreamWrite(SkyDomeClump, stream);

        RwStreamClose(stream, (void *)NULL);
    }
    else
    {
        RsErrorMessage(RWSTRING("Cannot open stream."));
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
InitializeMenu(void)
{
    static RwChar resolutionLabel[] = RWSTRING("Resolution");
    static RwChar tessellateLabel[] = RWSTRING("Tessellate_T");
    static RwChar exposureLabel[]   = RWSTRING("Exposure");
    static RwChar previewOnLabel[]  = RWSTRING("Preview_P");

    static RwChar skyColorsOnLabel[] = RWSTRING("Sky_S");
    static RwChar groundOnLabel[]    = RWSTRING("Ground_G");
    static RwChar wireMeshOnLabel[]  = RWSTRING("Wire mesh_W");
    static RwChar normalsOnLabel[]   = RWSTRING("Normals_N");

    static RwChar lensEffectOnLabel[] = RWSTRING("Lens effect_L");
    static RwChar cameraSpinOnLabel[] = RWSTRING("Camera spin_C");
    static RwChar saveDomeLabel[]     = RWSTRING("Save dome_X");

    static RwChar displayDataOnLabel[] = RWSTRING("Display data_D");
    static RwChar fpsOnLabel[]         = RWSTRING("FPS_F");

    if( RdMenuOpen(TRUE, &ForegroundColor, &BackgroundColor) )
    {
        RdMenuAddEntryInt(resolutionLabel, &TessellationRes, 
            (RdMenuTriggerCallBack)NULL, 1, 32, 1, (RwChar **)NULL);

        RdMenuAddEntryTrigger(tessellateLabel, TessellateCallback);

        RdMenuAddEntryReal(exposureLabel, &Exposure, 
            CurrentExposureCallback, 0.01f, 1.0f, 0.01f);

        RdMenuAddEntryBool(previewOnLabel, &PreviewOn, PreviewToggleCallback);

        RdMenuAddSeparator();

        RdMenuAddEntryBool(skyColorsOnLabel, &SkyColorsOn, SkyColorsToggleCallback);

        RdMenuAddEntryBool(groundOnLabel, &GroundOn, GroundToggleCallback);

        RdMenuAddEntryBool(wireMeshOnLabel, &WireMeshOn, (RdMenuTriggerCallBack)NULL);

        RdMenuAddEntryBool(normalsOnLabel, &NormalsOn, NormalsToggleCallback);

        RdMenuAddSeparator();

        RdMenuAddEntryBool(lensEffectOnLabel, &LensEffectOn, LensEffectToggleCallback);

        RdMenuAddEntryBool(cameraSpinOnLabel, &CameraSpinOn, CameraSpinToggleCallback);

        RdMenuAddEntryTrigger(saveDomeLabel, SkyDomeSaveCallback);

        RdMenuAddSeparator();

        RdMenuAddEntryBool(displayDataOnLabel, &DisplayDataOn, (RdMenuTriggerCallBack)NULL);

        RdMenuAddEntryBool(fpsOnLabel, &FPSOn, (RdMenuTriggerCallBack)NULL);

        return TRUE;
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static RwBool 
Initialize3D(void *param)
{
    RwV3d yAxis = {0.0f, 1.0f, 0.0f};
    RwV3d xAxis = {1.0f, 0.0f, 0.0f};

    if( !RsRwInitialize(param) )
    {
        RsErrorMessage(RWSTRING("Error initializing RenderWare."));

        return FALSE;
    }

    Charset = RtCharsetCreate(&ForegroundColor, &BackgroundColor);
    if( (RtCharset *)NULL == Charset )
    {
        RsErrorMessage(RWSTRING("Cannot create raster chraset."));
    
        return FALSE;
    }

    World = CreateWorld();
    if( (RpWorld *)NULL == World )
    {
        RsErrorMessage(RWSTRING("Cannot create world."));

        return FALSE;
    }

    Camera = CreateCamera(World);
    if( (RwCamera *)NULL == Camera )
    {
        RsErrorMessage(RWSTRING("Cannot create camera."));

        return FALSE;
    }
    else
    {
        /*
         * Setup the position and orientation of the camera
         * for the initial view...
         */
        RwFrameSetIdentity(RwCameraGetFrame(Camera));

        /*
         * Initialize the camera's LTM for the view from
         * inside the skydome...
         */
        RwMatrixRotate(&CameraInsideLTM, &xAxis, 
            -CameraElevation, rwCOMBINEREPLACE);
        
        RwMatrixRotate(&CameraInsideLTM, &yAxis, 
            CameraAzimuth, rwCOMBINEPOSTCONCAT);

        CameraUpdateHFoV(0.0f);
    }

    if( !InitializeColor() )
    {
        RsErrorMessage(RWSTRING("Error initializing color."));

        return FALSE;
    }

    SkyDomeClump = CreateClump(World);
    if( (RpClump *)NULL == SkyDomeClump )
    {
        RsErrorMessage(RWSTRING("Cannot create skydome clump."));

        return FALSE;
    }
    else
    {
        RwFrame *clumpFrame;
        RwV3d outsidePos = {0.0f, 0.0f, 4.5f};

        /*
         * Setup the position and orientation of the skydome
         * for the initial view...
         */
        clumpFrame = RpClumpGetFrame(SkyDomeClump);
        RwFrameRotate(clumpFrame, &yAxis, 180.0f, rwCOMBINEREPLACE);
        RwFrameRotate(clumpFrame, &xAxis, -45.0f, rwCOMBINEPOSTCONCAT);
        RwFrameTranslate(clumpFrame, &outsidePos, rwCOMBINEPOSTCONCAT);

        /*
         * Initialize the skydome's LTM for the view from inside
         * the dome...
         */
        RwMatrixSetIdentity(&SkyDomeInsideLTM);

        /*
         * Position the Sun - this will automatically produce the first sky
         * color calculations...
         */
        SkyDomeRotateSun(50.0f, 20.0f);

        /*
         * Update the skydome-space position and at-vector of the camera
         * (for wire-frame culling purposes)...
         */
        UpdateClumpSpaceCameraParameters();

        ElRes = TessellationRes;
        AzRes = ElRes * 4;
    }

    GroundTex = CreateGroundTexture();
    if( (RwTexture *)NULL == GroundTex )
    {
        RsErrorMessage(RWSTRING("Cannot create ground texture."));

        return FALSE;
    }

    Pointer = RdCameraCreateCrossHair();
    if( (RwRaster *)NULL == Pointer )
    {
        RsErrorMessage(RWSTRING("Cannot create camera pointer."));

        return FALSE;
    }

    if( !InitializeMenu() )
    {
        RsErrorMessage(RWSTRING("Error initializing menu."));

        return FALSE;
    }

    if( !InitializeFlares() )
    {
        RsErrorMessage(RWSTRING("Error initializing camera flares."));

        return FALSE;
    }

#ifdef RWMETRICS
    RsMetricsOpen(Camera);
#endif /* RWMETRICS */

    return TRUE;
}


/*
 *****************************************************************************
 */
static void 
Terminate3D(void)
{
    DestroyIm3DBuffers();

#ifdef RWMETRICS
    RsMetricsClose();
#endif /* RWMETRICS */

    TerminateFlares();

    RdMenuClose();

    if( (RwRaster *)NULL != Pointer )
    {
        RwRasterDestroy(Pointer);
    }

    if( (RwTexture *)NULL != GroundTex )
    {
        RwTextureDestroy(GroundTex);
    }

    if( (RpClump *)NULL != SkyDomeClump )
    {
        RpWorldRemoveClump(World, SkyDomeClump);

        RpClumpDestroy(SkyDomeClump);
    }

    SkyDomeDestroySkyColorBuffer();

    if( (RwCamera *)NULL != Camera )
    {
        RpWorldRemoveCamera(World, Camera);

        RdCameraDestroy(Camera);
    }

    if( (RpWorld *)NULL != World ) 
    {
        RpWorldDestroy(World);
    }

    if( (RtCharset *)NULL != Charset )
    {
        RwRasterDestroy(Charset);
    }

    RsRwTerminate();

    return;
}


/*
 *****************************************************************************
 */
static RwBool 
AttachPlugins(void)
{
    /* 
     * Attach world plug-in...
     */
    if( !RpWorldPluginAttach() )
    {
        return FALSE;
    }

#ifdef RWLOGO
    /* 
     * Attach logo plug-in...
     */
    if( !RpLogoPluginAttach() )
    {
        return FALSE;
    }
#endif

    return TRUE;
}


/*
 *****************************************************************************
 */
static void 
Idle(void)
{
    RwUInt32 thisTime;
    RwReal deltaTime;

    static RwBool firstCall = TRUE;
    static RwUInt32 lastFrameTime, lastAnimTime;

    if( firstCall )
    {
        lastFrameTime = RsTimer();

        firstCall = FALSE;
    }

    thisTime = RsTimer();

    /* 
     * Has a second elapsed since we last updated the FPS...
     */
    if( thisTime > (lastFrameTime + 1000) )
    {
        /* 
         * Capture the frame counter...
         */
        FramesPerSecond = FrameCounter;
        
        /*
         * ...and reset...
         */
        FrameCounter = 0;
        
        lastFrameTime = thisTime;
    }

    if( CameraSpinOn )
    {
        deltaTime = (thisTime - lastAnimTime) * 0.001f;

        CameraPoint(0.0f, SPINRATE * deltaTime);
    }

    lastAnimTime = thisTime;

    Render();

    return;
}


/*
 *****************************************************************************
 */
RsEventStatus
AppEventHandler(RsEvent event, void *param)
{
    switch( event )
    {
        case rsINITIALIZE:
        {
            return Initialize() ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsCAMERASIZE:
        {
            RdCameraSize(Camera, (RwRect *)param, 
                CameraViewWindow, DEFAULT_ASPECTRATIO);

            return rsEVENTPROCESSED;
        }

        case rsRWINITIALIZE:
        {
            return Initialize3D(param) ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsRWTERMINATE:
        {
            Terminate3D();

            return rsEVENTPROCESSED;
        }

        case rsPLUGINATTACH:
        {
            return AttachPlugins() ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsINPUTDEVICEATTACH:
        {
            AttachInputDevices();

            return rsEVENTPROCESSED;
        }

        case rsIDLE:
        {
            Idle();

            return rsEVENTPROCESSED;
        }

        default:
        {
            return rsEVENTNOTPROCESSED;
        }
    }
}

/*
 *****************************************************************************
 */
