
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 2000, 2001 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * geometry.c
 *
 * Copyright (C) 2000, 2001 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: Sky dome creator for RW3.
 *
 ****************************************************************************/

#include "rwcore.h"
#include "rpworld.h"

#include "skeleton.h"

#include "main.h"

static RwRGBA Black   = {  0,   0,   0, 255};
static RwRGBA SkyBlue = { 50, 100, 255, 255};
static RwRGBA White   = {255, 255, 255, 255};



/*
 *****************************************************************************
 */
RpGeometry *
SkyDomeCreateGeometry(RwInt32 tessellationRes)
{
    RwInt32 i, j;
    RwInt32 azRes, elRes;
    RwInt32 poleV0, poleV1, poleV2;
    RwInt32 numVertices, numTriangles;
    RwReal delta;
    RpGeometry *geometry;
    RpMorphTarget *morphTarget;
    RwSphere boundingSphere;
    RwV3d *vlist, *nlist;
    RwRGBA *prelight;
    RpTriangle *tlist;
    RpMaterial *material;

    elRes = tessellationRes;
    azRes = elRes * 4;
    delta = 2.0f * rwPI / azRes;

    numVertices = azRes * elRes + 1;
    numTriangles = azRes * (elRes - 1) * 2 + azRes;

    geometry = RpGeometryCreate(numVertices, numTriangles, 
        rpGEOMETRYPRELIT | rpGEOMETRYNORMALS);

    /*
     * Calculate the vertex positions and normals...
     */
    morphTarget = RpGeometryGetMorphTarget(geometry, 0);
    vlist = RpMorphTargetGetVertices(morphTarget);
    nlist = RpMorphTargetGetVertexNormals(morphTarget);
    prelight = RpGeometryGetPreLightColors(geometry);

    for(j=0; j<elRes; j++)
    {
        RwReal cosEl, sinEl;

        cosEl = (RwReal)RwCos(j * delta);
        sinEl = (RwReal)RwSin(j * delta);

        for(i=0; i<azRes; i++)
        {
            RwReal cosAz, sinAz;

            cosAz = (RwReal)RwCos(i * delta);
            sinAz = (RwReal)RwSin(i * delta);

            vlist->x = cosEl * sinAz;
            vlist->y = sinEl;
            vlist->z = cosEl * cosAz;

            *nlist = *vlist;

            *prelight = Black;

            vlist++;
            nlist++;
            prelight++;
        }
    }

    /*
     * Last vertex is the pole...
     */
    vlist->x = vlist->z = 0.0f;
    vlist->y = 1.0f;
    *nlist = *vlist;
    *prelight = Black;

    /*
     * Set the triangle loops and material...
     */
    material = RpMaterialCreate();
    RpMaterialSetColor(material, &White);
    RpMaterialSetTexture(material, (RwTexture *)NULL);

    tlist = RpGeometryGetTriangles(geometry);

    for(j=0; j<elRes-1; j++)
    {
        RwInt32 linkV0, linkV1, linkV2, linkV3;

        for(i=0; i<azRes-1; i++)
        {
            RwInt32 v0, v1, v2, v3;

            v0 = i + j * azRes;
            v1 = v0 + 1;
            v2 = i + (j + 1) * azRes;
            v3 = v2 + 1;

            if( 0 == i )
            {
                linkV1 = v0;
                linkV3 = v2;
            }
            else if( azRes - 2 == i )
            {
                linkV0 = v1;
                linkV2 = v3;
            }

            RpGeometryTriangleSetVertexIndices(geometry, tlist, 
                (RwUInt16)v0, (RwUInt16)v3, (RwUInt16)v2);
            RpGeometryTriangleSetMaterial(geometry, tlist, material);

            RpGeometryTriangleSetVertexIndices(geometry, tlist + 1, 
                (RwUInt16)v0, (RwUInt16)v1, (RwUInt16)v3);
            RpGeometryTriangleSetMaterial(geometry, tlist + 1, material);

            tlist += 2;
        }

        /*
         * Define a pair of horizontal link triangles...
         */
        RpGeometryTriangleSetVertexIndices(geometry, tlist, 
            (RwUInt16)linkV0, (RwUInt16)linkV3, (RwUInt16)linkV2);
        RpGeometryTriangleSetMaterial(geometry, tlist, material);

        RpGeometryTriangleSetVertexIndices(geometry, tlist + 1, 
            (RwUInt16)linkV0, (RwUInt16)linkV1, (RwUInt16)linkV3);
        RpGeometryTriangleSetMaterial(geometry, tlist + 1, material);

        tlist += 2;
    }

    /*
     * Define the pole triangles...
     */
    poleV0 = azRes * (elRes - 1);
    poleV1 = poleV0 + 1;
    poleV2 = numVertices - 1;

    for(i=0; i<azRes-1; i++)
    {
        RpGeometryTriangleSetVertexIndices(geometry, tlist, 
            (RwUInt16)poleV0, (RwUInt16)poleV1, (RwUInt16)poleV2);
        RpGeometryTriangleSetMaterial(geometry, tlist, material);

        poleV0++;
        poleV1++;
        tlist++;
    }

    /*
     * ...and finally a pole link triangle...
     */
    poleV0 = azRes * (elRes - 1);
    poleV1 = poleV0 + azRes - 1;
    poleV2 = numVertices - 1;
    RpGeometryTriangleSetVertexIndices(geometry, tlist, 
            (RwUInt16)poleV1, (RwUInt16)poleV0, (RwUInt16)poleV2);
    RpGeometryTriangleSetMaterial(geometry, tlist, material);

    RpMorphTargetCalcBoundingSphere(morphTarget, &boundingSphere);
    RpMorphTargetSetBoundingSphere(morphTarget, &boundingSphere);
    RpGeometryUnlock(geometry);

    /*
     * Remove the application's ownership of the material now that the 
     * geometry and geometry's triangles have it...
     */
    RpMaterialDestroy(material);

    return geometry;
}


/*
 *****************************************************************************
 */
void 
SkyDomeInvertGeometry(void)
{
    RpGeometry *geometry;

    geometry = RpAtomicGetGeometry(SkyDomeAtomic);
    if( (RpGeometry *)NULL != geometry )
    {
        RwInt32 numVert, numTri, i;
        RpMorphTarget *morphTarget;
        RwV3d *normal;
        RpTriangle *triangle;

        RpGeometryLock(geometry, rpGEOMETRYLOCKNORMALS|rpGEOMETRYLOCKPOLYGONS);

        numVert = RpGeometryGetNumVertices(geometry);
        numTri = RpGeometryGetNumTriangles(geometry);

        morphTarget = RpGeometryGetMorphTarget(geometry, 0);
        normal = RpMorphTargetGetVertexNormals(morphTarget);

        for(i=0; i<numVert; i++)
        {
            RwV3dScale(normal, normal, -1.0f);

            normal++;
        }

        triangle = RpGeometryGetTriangles(geometry);

        for(i=0; i<numTri; i++)
        {
            RwUInt16 vert1, vert2, vert3;

            RpGeometryTriangleGetVertexIndices(geometry, triangle,
                &vert1, &vert2, &vert3);

            RpGeometryTriangleSetVertexIndices(geometry, triangle, 
                vert1, vert3, vert2);

            triangle++;
        }

        RpGeometryUnlock(geometry);
    }

    return;
}

/*
 *****************************************************************************
 */
