
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 2000, 2001 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * flare.c
 *
 * Copyright (C) 2000, 2001 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: Sky dome creator for RW3.
 *
 ****************************************************************************/

#include "rwcore.h"

#include "main.h"
#include "skycolor.h"

#define NUMTEXTURES (8)
#define NUMFLARES (12)

typedef struct
{
    RwInt32 texIndex;
    RwReal axisLoc;
    RwReal scale;
    RwReal opacity;
    RwV3d position;
}
Flare;

static RwTexture *FlareTextures[NUMTEXTURES];

static Flare Flares[NUMFLARES] = 
{
    {2,  1.30f, 0.06f, 0.6f},
    {3,  1.00f, 0.15f, 0.4f},
    {1,  0.50f, 0.30f, 0.3f},
    {3,  0.20f, 0.07f, 0.3f},
    {0,  0.05f, 0.06f, 0.3f},
    {4, -0.25f, 0.11f, 0.5f},
    {4, -0.40f, 0.03f, 0.6f},
    {4, -0.60f, 0.06f, 0.4f},
    {4, -1.00f, 0.04f, 0.2f},
    {5,  1.00f, 0.45f, 0.4f},
    {6,  1.00f, 0.30f, 0.4f},
    {7,  1.00f, 0.45f, 0.4f}
};

static RwV3d FlareAxis;
static RwV3d FlareNormal;

static RwTexCoords QuadUVs[4] = 
{
    { 0.0f, 0.0f },
    { 1.0f, 0.0f },
    { 1.0f, 1.0f },
    { 0.0f, 1.0f }
};



/*
 *****************************************************************************
 */
RwBool 
InitializeFlares(void)
{
    RwInt32 i;

    for(i=0; i<NUMTEXTURES; i++)
    {
        RwChar name[32], mask[32];

        rwsprintf(name, RWSTRING("flare%d"), i);
        rwsprintf(mask, RWSTRING("flare%dm"), i);

        FlareTextures[i] = RwTextureRead(name, mask);

        if( (RwTexture *)NULL == FlareTextures[i] )
        {
            return FALSE;
        }
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
RwBool 
TerminateFlares(void)
{
    RwInt32 i;

    for(i=0; i<NUMTEXTURES; i++)
    {
        RwTextureDestroy(FlareTextures[i]);
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
RwBool 
UpdateFlares(void)
{
    RwV3d *cameraAt, *cameraPos;
    RwV3d newSunPos;
    RwReal dot;
    RwV3d flareCenter, flareAxis;
    RwInt32 i;

    /*
     * First determine if the Sun is behind the camera. If so
     * no flares are visible...
     */
    cameraAt = RwMatrixGetAt(RwFrameGetLTM(RwCameraGetFrame(Camera)));

    dot = RwV3dDotProduct(&SunPos, cameraAt);

    if( dot < 0.0f )
    {
        return FALSE;
    }

    /*
     * Project the Sun's position in front of the camera's view plane...
     */
    cameraPos = RwMatrixGetPos(RwFrameGetLTM(RwCameraGetFrame(Camera)));

    RwV3dScale(&newSunPos, &SunPos, 1.01f / dot);
    RwV3dAdd(&newSunPos, &newSunPos, cameraPos);

    /*
     * Calculate the flare center and axis...
     */
    RwV3dScale(&flareCenter, cameraAt, 1.01f);
    RwV3dAdd(&flareCenter, &flareCenter, cameraPos);

    RwV3dSub(&flareAxis, &newSunPos, &flareCenter);

    /*
     * Finally, calculate the 3D position of each flare
     * along the flare axis...
     */
    for(i=0; i<NUMFLARES; i++)
    {
        RwV3dScale(&Flares[i].position, &flareAxis, Flares[i].axisLoc);
        RwV3dAdd(&Flares[i].position, &Flares[i].position, &flareCenter);
    }

    RwV3dNormalize(&FlareAxis, &flareAxis);
    RwV3dCrossProduct(&FlareNormal, &FlareAxis, cameraAt);

    return TRUE;
}


/*
 *****************************************************************************
 */
RwBool 
RenderFlares(void)
{
    RwInt32 i;

    for(i=0; i<NUMFLARES; i++)
    {
        RwV3d sx, sy, quad[4], *pos;
        RwIm3DVertex im3DVertices[6];
        RwInt32 vert;
        RwRGBA color = {128, 64, 64, 255};

        color.red = (RwUInt8)(color.red * Flares[i].opacity);
        color.green = (RwUInt8)(color.green * Flares[i].opacity);
        color.blue = (RwUInt8)(color.blue * Flares[i].opacity);

        /*
         * Positions of the quad vertices...
         */
        RwV3dScale(&sx, &FlareAxis, Flares[i].scale);
        RwV3dScale(&sy, &FlareNormal, Flares[i].scale);

        RwV3dAdd(&quad[0], &Flares[i].position, &sx);
        RwV3dAdd(&quad[0], &quad[0], &sy);

        RwV3dSub(&quad[1], &Flares[i].position, &sx);
        RwV3dAdd(&quad[1], &quad[1], &sy);

        RwV3dSub(&quad[2], &Flares[i].position, &sx);
        RwV3dSub(&quad[2], &quad[2], &sy);

        RwV3dAdd(&quad[3], &Flares[i].position, &sx);
        RwV3dSub(&quad[3], &quad[3], &sy);

        /*
         * First immediate mode triangle...
         */
        vert = 0;
        pos = &quad[vert];
        RwIm3DVertexSetPos(&im3DVertices[0], pos->x, pos->y, pos->z);
        RwIm3DVertexSetRGBA(&im3DVertices[0], 
            color.red, color.green, color.blue, color.alpha);
        RwIm3DVertexSetU(&im3DVertices[0], QuadUVs[vert].u);
        RwIm3DVertexSetV(&im3DVertices[0], QuadUVs[vert].v);

        vert = 1;
        pos = &quad[vert];
        RwIm3DVertexSetPos(&im3DVertices[1], pos->x, pos->y, pos->z);
        RwIm3DVertexSetRGBA(&im3DVertices[1], 
            color.red, color.green, color.blue, color.alpha);
        RwIm3DVertexSetU(&im3DVertices[1], QuadUVs[vert].u);
        RwIm3DVertexSetV(&im3DVertices[1], QuadUVs[vert].v);

        vert = 2;
        pos = &quad[vert];
        RwIm3DVertexSetPos(&im3DVertices[2], pos->x, pos->y, pos->z);
        RwIm3DVertexSetRGBA(&im3DVertices[2], 
            color.red, color.green, color.blue, color.alpha);
        RwIm3DVertexSetU(&im3DVertices[2], QuadUVs[vert].u);
        RwIm3DVertexSetV(&im3DVertices[2], QuadUVs[vert].v);

        /*
         * Second immediate mode triangle...
         */
        vert = 0;
        pos = &quad[vert];
        RwIm3DVertexSetPos(&im3DVertices[3], pos->x, pos->y, pos->z);
        RwIm3DVertexSetRGBA(&im3DVertices[3], 
            color.red, color.green, color.blue, color.alpha);
        RwIm3DVertexSetU(&im3DVertices[3], QuadUVs[vert].u);
        RwIm3DVertexSetV(&im3DVertices[3], QuadUVs[vert].v);

        vert = 2;
        pos = &quad[vert];
        RwIm3DVertexSetPos(&im3DVertices[4], pos->x, pos->y, pos->z);
        RwIm3DVertexSetRGBA(&im3DVertices[4], 
            color.red, color.green, color.blue, color.alpha);
        RwIm3DVertexSetU(&im3DVertices[4], QuadUVs[vert].u);
        RwIm3DVertexSetV(&im3DVertices[4], QuadUVs[vert].v);

        vert = 3;
        pos = &quad[vert];
        RwIm3DVertexSetPos(&im3DVertices[5], pos->x, pos->y, pos->z);
        RwIm3DVertexSetRGBA(&im3DVertices[5], 
            color.red, color.green, color.blue, color.alpha);
        RwIm3DVertexSetU(&im3DVertices[5], QuadUVs[vert].u);
        RwIm3DVertexSetV(&im3DVertices[5], QuadUVs[vert].v);

        RwRenderStateSet(rwRENDERSTATETEXTURERASTER, 
            RwTextureGetRaster(FlareTextures[Flares[i].texIndex]));

        /*
         * Render the quad...
         */
        if( RwIm3DTransform(im3DVertices, 6, (RwMatrix *)NULL, rwIM3D_VERTEXUV) != (void *)NULL)
        {
            RwIm3DRenderPrimitive(rwPRIMTYPETRILIST);

            RwIm3DEnd();
        }
    }

    return TRUE;
}

/*
 *****************************************************************************
 */

