
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 1999, 2000 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * play.c
 *
 * Copyright (C) 1999, 2000 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: To illustrate collision detection and rigid-body dynamics.
 *
 ****************************************************************************/

#include "rwcore.h"
#include "rpworld.h"
#include "rpcollis.h"
#include "rtpick.h"

#include "main.h"
#include "dice.h"

typedef struct
{
    RpIntersection intersection;
    RpAtomic *pickedAtomic;
    RwReal minDistance;
}
AtomicIntersectParams;

RwInt32 PickedDieValues[NUMDICE];

RwInt32 CurrentDieValue;

RwBool DiceRolling = FALSE;
RwInt32 RollNumber = -1;
RwInt32 NumInactiveDice;

RwBool CurrentDieOn = FALSE;

RwChar RollCaption[128] =
    RWSTRING("First Roll (right-click to roll, left-click to pick)");

RwBool DiePreview = FALSE;
RwInt32 DiePreviewX;
RwInt32 DiePreviewY;



/*
 *****************************************************************************
 */
RwBool 
DiceRoll(void)
{
    if( !DiceRolling )
    {
        RollNumber = (++RollNumber) % 3;

        if( 0 == RollNumber )
        {
            RwInt32 i;

            for(i=0; i<NUMDICE; i++ )
            {
                RpAtomicSetFlags(Dice[i].atomic, 
                    rpATOMICRENDER | rpATOMICCOLLISIONTEST);

                Dice[i].active = TRUE;
            }

            NumInactiveDice = 0;

            EnergyThreshold = MINENERGY;

            rwstrcpy(RollCaption, RWSTRING("First Roll..."));
        }
        else if( 1 == RollNumber )
        {
            rwstrcpy(RollCaption, RWSTRING("Second Roll..."));
        }
        else if( 2 == RollNumber )
        {
            rwstrcpy(RollCaption, RWSTRING("Third Roll..."));
        }

        InitializeDice();

        DiceRolling = TRUE;

        CurrentDieOn = FALSE;
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static RwInt32  
DieValue(RwInt32 diceNumber)
{
    RwMatrix *matrix;

    /*
     * Assumes world 'up' is the positive y-axis, opposite the direction
     * of gravity, and the texture maps will not be swapped!...
     */

    matrix = RwFrameGetMatrix(RpAtomicGetFrame(Dice[diceNumber].atomic));

    if( RwMatrixGetRight(matrix)->y > 0.5f )
    {
        return 5;
    }
    else if( RwMatrixGetRight(matrix)->y < -0.5f )
    {
        return 2;
    }
    else if( RwMatrixGetUp(matrix)->y > 0.5f )
    {
        return 6;
    }
    else if( RwMatrixGetUp(matrix)->y < -0.5f )
    {
        return 1;
    }
    else if( RwMatrixGetAt(matrix)->y > 0.5f )
    {
        return 3;
    }
    else if( RwMatrixGetAt(matrix)->y < -0.5f )
    {
        return 4;
    }
    else
    {
        return 0;
    }
}


/*
 *****************************************************************************
 */
static RpCollisionTriangle * 
TriangleIntersectionCallback(RpIntersection *intersection, 
           RpCollisionTriangle *triangle, RwReal distance, void *data)
{
    if( distance < *(RwReal *)data )
    {
        *(RwReal *)data = distance;
    }

    return triangle;
}


/*
 *****************************************************************************
 */
static RpAtomic * 
AtomicIntersectLine(RpIntersection *intersection, RpWorldSector *sector, 
        RpAtomic *atomic, RwReal distance, void *data)
{
    AtomicIntersectParams *intersectParams;
    RwReal oldDistance;

    intersectParams = (AtomicIntersectParams *)data;

    oldDistance = intersectParams->minDistance;

    RpAtomicForAllIntersections(atomic, &intersectParams->intersection, 
        TriangleIntersectionCallback, &intersectParams->minDistance);
    
    if( intersectParams->minDistance < oldDistance )
    {
        intersectParams->pickedAtomic = atomic;
    }

    return atomic;
}



/*
 *****************************************************************************
 */
static RpAtomic * 
CameraPickAtomicOnPixel(RwCamera *camera, RwInt32 screenX, RwInt32 screenY)
{
    RwV2d pixel;
    AtomicIntersectParams intersectParams;

    pixel.x = (RwReal)screenX;
    pixel.y = (RwReal)screenY;

    intersectParams.intersection.type = rpINTERSECTLINE;

    RwCameraCalcPixelRay(camera, &intersectParams.intersection.t.line, &pixel);

    intersectParams.pickedAtomic = (RpAtomic *)NULL;
    intersectParams.minDistance = RwRealMAXVAL;

    RpWorldForAllAtomicIntersections(World, &intersectParams.intersection, 
        AtomicIntersectLine, &intersectParams);

    return intersectParams.pickedAtomic;
}


/*
 *****************************************************************************
 */
void 
DicePreview(RwInt32 x, RwInt32 y)
{
    if( !DiceRolling )
    {
        RpAtomic *pickedAtomic;

        pickedAtomic = CameraPickAtomicOnPixel(Camera, x, y);

        if( (RpAtomic *)NULL != pickedAtomic )
        {
            RwInt32 i, dieValue;

            for(i=0; i<NUMDICE; i++)
            {
                if( pickedAtomic == Dice[i].atomic )
                {
                    dieValue = DieValue(i);

                    if( 0 != dieValue )
                    {
                        CurrentDieValue = dieValue;
                        DiePreviewX = x;
                        DiePreviewY = y;

                        CurrentDieOn = TRUE;
                    }

                    break;
                }
            }
        }
        else
        {
            CurrentDieOn = FALSE;
        }
    }

    return;
}



/*
 *****************************************************************************
 */
void 
DiceSelect(RwInt32 x, RwInt32 y)
{
    if( !DiceRolling )
    {
        RpAtomic *pickedAtomic;

        pickedAtomic = CameraPickAtomicOnPixel(Camera, x, y);

        if( (RpAtomic *)NULL != pickedAtomic )
        {
            RwInt32 i, dieValue;

            for(i=0; i<NUMDICE; i++)
            {
                if( pickedAtomic == Dice[i].atomic )
                {
                    dieValue = DieValue(i);

                    if( 0 != dieValue )
                    {
                        PickedDieValues[NumInactiveDice] = dieValue;

                        Dice[i].active = FALSE;

                        RpAtomicSetFlags(Dice[i].atomic, 0);

                        /* 
                         * Cause the frame to be updated so the flags
                         * change is reflected in the sector ties.
                         */
                        RwFrameUpdateObjects(RpAtomicGetFrame(Dice[i].atomic));

                        NumInactiveDice++;

                        if( NUMDICE == NumInactiveDice )
                        {
                            RwInt32 i;

                            for(i=0; i<NUMDICE; i++)
                            {
                                Dice[i].active = TRUE;

                                RpAtomicSetFlags(Dice[i].atomic, rpATOMICRENDER);
                            }

                            ResetDiceOnTable();

                            RollNumber = 2;
                            rwstrcpy(RollCaption, RWSTRING("Game Over"));
                        }
                        else
                        {
                            EnergyThreshold = MINENERGY * 
                                (RwReal)(NUMDICE - NumInactiveDice) / NUMDICE;
                        }

                        CurrentDieOn = FALSE;
                    }

                    break;
                }
            }
        }
    }

    return;
}


/*
 *****************************************************************************
 */
void 
GatherRemainingDieValues(void)
{
    RwInt32 i;

    for(i=0; i<NUMDICE; i++)
    {
        if( Dice[i].active )
        {
            RwInt32 dieValue;

            dieValue = DieValue(i);

            PickedDieValues[NumInactiveDice] = dieValue;

            NumInactiveDice++;

            /*
             * Turn-off picking...
             */
            RpAtomicSetFlags(Dice[i].atomic, rpATOMICRENDER);

            /* 
             * Cause the frame to be updated so the flags
             * change is reflected in the world-sector ties...
             */
            RwFrameUpdateObjects(RpAtomicGetFrame(Dice[i].atomic));
        }
    }

    return;
}

/*
 *****************************************************************************
 */
