
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 1999, 2000 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * main.c
 *
 * Copyright (C) 1999, 2000 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: To illustrate collision detection and rigid-body dynamics.
 *
 ****************************************************************************/

#include "rwcore.h"
#include "rpworld.h"
#include "rpcollis.h"
#include "rprandom.h"
#include "rpmatfx.h"

#ifdef RWLOGO
#include "rplogo.h"
#endif

#include "rtcharse.h"

#include "skeleton.h"
#include "menu.h"
#include "camera.h"

#ifdef RWMETRICS
#include "metrics.h"
#endif

#include "main.h"
#include "dice.h"
#include "play.h"
#include "events.h"

#define DEFAULT_SCREEN_WIDTH (640)
#define DEFAULT_SCREEN_HEIGHT (480)
#define DEFAULT_ASPECT_RATIO (4.0f/3.0f)

#define INIT_HFOV (60.0f)

#define DEG_TO_RAD(x) ((x) * rwPI / 180.0f)

#define RESAMPLED_SIZE (64)

typedef struct
{
    RwReal tilt;
    RwReal turn;
    RwReal distance;
    RwReal hfov;
}
CameraView;

static RwBool FPSOn = FALSE;

static RwInt32 FrameCounter = 0;
static RwInt32 FramesPerSecond = 0;

static RwRGBA ForegroundColor = {200, 200, 200, 255};
static RwRGBA BackgroundColor = { 64,  64,  64,   0};

static RtCharset *Charset = (RtCharset *)NULL;
static RpLight *AmbientLight = (RpLight *)NULL;
static RpLight *MainLight = (RpLight *)NULL;
static RwRaster *Pointer = (RwRaster *)NULL;

static RwRaster *DieFaceRasters[6];

static CameraView PresetViews[] = 
{
    {-51.0f,   9.0f, 27.00f, INIT_HFOV},
    {-89.0f,  45.0f, 30.00f, INIT_HFOV},
    {-21.0f, 116.0f, 11.00f, INIT_HFOV},
    {-12.0f, -45.0f, 13.00f, INIT_HFOV}
};

static CameraView CurrentView;

static RwInt32 CurrentViewIndex;
static RwInt32 NumViews = (RwInt32)(sizeof(PresetViews)/sizeof(CameraView));
static RwReal CurrentViewWindow;

static RwTexture *EnvMap = (RwTexture *)NULL;
static RwBool EnvMapOn = FALSE;

RpWorld *World = (RpWorld *)NULL;
RwCamera *Camera = (RwCamera *)NULL;



/*
 *****************************************************************************
 */
static void 
CameraSetView(RwCamera *camera, CameraView view)
{
    RwFrame *frame;
    RwV3d at;
    RwV3d xAxis = {1.0f, 0.0f, 0.0f};
    RwV3d yAxis = {0.0f, 1.0f, 0.0f};

    /*
     * Set the field-of-view.
     * Note: this is handled in the camera resize code so we simply force 
     * a resize event...
     */
    CurrentViewWindow = (RwReal)RwTan(DEG_TO_RAD(view.hfov * 0.5f));
    RsEventHandler(rsCAMERASIZE, (void *)NULL);

    frame = RwCameraGetFrame(camera);

    /*
     * Orient the camera...
     */
    RwFrameRotate(frame, &xAxis, -view.tilt, rwCOMBINEREPLACE);
    RwFrameRotate(frame, &yAxis, view.turn, rwCOMBINEPOSTCONCAT);

    /*
     * Position the camera...
     */
    at = *RwMatrixGetAt(RwFrameGetMatrix(frame));
    RwV3dScale(&at, &at, -view.distance);
    RwV3dAdd(&at, &at, RpWorldGetOrigin(World));
    RwFrameTranslate(frame, &at, rwCOMBINEPOSTCONCAT);

    return;
}


/*
 *****************************************************************************
 */
void 
CameraPoint(RwReal tilt, RwReal turn)
{
    RwFrame *frame;
    RwV3d pos;
    RwV3d xAxis = {1.0f, 0.0f, 0.0f};
    RwV3d yAxis = {0.0f, 1.0f, 0.0f};

    CurrentView.tilt += tilt;
    CurrentView.turn += turn;

    /*
     * Limit the camera's elevation so that it never quite reaches
     * exactly +90 or -90 degrees to avoid the singularity in these
     * situations...
     */
    if( CurrentView.tilt > 89.0f )
    {
        CurrentView.tilt = 89.0f;
    }
    else if( CurrentView.tilt < -89.0f )
    {
        CurrentView.tilt = -89.0f;
    }

    /*
     * Keep the azimuth in the range -180 to +180 degrees...
     */
    if( CurrentView.turn > 180.0f )
    {
        CurrentView.turn -= 360.0f;
    }
    else if( CurrentView.turn < -180.0f )
    {
        CurrentView.turn += 360.0f;
    }

    frame = RwCameraGetFrame(Camera);

    /*
     * Remember where the camera is...
     */
    pos = *RwMatrixGetPos(RwFrameGetMatrix(frame));

    /*
     * Reset the camera's frame; we're dealing with absolute
     * orientations here...
     */
    RwFrameRotate(frame, &xAxis, -CurrentView.tilt, rwCOMBINEREPLACE);
    RwFrameRotate(frame, &yAxis, CurrentView.turn, rwCOMBINEPOSTCONCAT);

    /*
     * Put the camera back to where it was...
     */
    RwFrameTranslate(frame, &pos, rwCOMBINEPOSTCONCAT);

    return;
}


/*
 *****************************************************************************
 */
void
CameraTranslateZ(RwReal dist)
{
    RwFrame *frame;
    RwV3d at, pos;

    /*
     * Move the camera along its look-at vector the given distance...
     */
    frame = RwCameraGetFrame(Camera);

    at = *RwMatrixGetAt(RwFrameGetMatrix(frame));

    RwV3dScale(&at, &at, dist);

    RwFrameTranslate(frame, &at, rwCOMBINEPOSTCONCAT);

    /*
     * Update the camera view distance parameter...
     */
    pos = *RwMatrixGetPos(RwFrameGetMatrix(frame));
    RwV3dSub(&pos, &pos, RpWorldGetOrigin(World));
    CurrentView.distance = RwV3dLength(&pos);

    return;
}


/*
 *****************************************************************************
 */
static RpMaterial * 
TextureSetFilterMode(RpMaterial *material, void *data)
{
    RwTextureFilterMode filterMode;
    RwTexture *texture;

    filterMode = *(RwTextureFilterMode *)data;

    texture = RpMaterialGetTexture(material);
    if( (RwTexture *)NULL != texture ) 
    {
        RwTextureSetFilterMode(texture, filterMode);
    }

    return material;
}


/*
 *****************************************************************************
 */
static RpWorld *
CreateWorld(void)
{
    RwChar *path;
    RwStream *stream;
    RpWorld *world = (RpWorld *)NULL;

    path = RsPathnameCreate(RWSTRING("models/dicetabl.bsp"));

    /*
     * Read the table BSP from a binary format file...
     */
    RsSetModelTexturePath(path);

    RwImageSetGamma(1.4f);

    stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMREAD, path);
    if( (RwStream *)NULL != stream )
    {
        if( RwStreamFindChunk(stream, rwID_WORLD, (RwUInt32 *)NULL, (RwUInt32 *)NULL) ) 
        {
            world = RpWorldStreamRead(stream);
        }

        RwStreamClose(stream, (void *)NULL);
    }

    RwImageSetGamma(1.0f);

    RsPathnameDestroy(path);

    return world;
}


/*
 *****************************************************************************
 */
static RpLight *
CreateAmbientLight(RpWorld *world)
{
    RpLight *light;

    light = RpLightCreate(rpLIGHTAMBIENT);
    if( (RpLight *)NULL != light )
    {
        RwRGBAReal color = {0.3f, 0.3f, 0.3f, 1.0f};

        RpLightSetColor(light, &color);

        RpWorldAddLight(world, light);

        return light;
    }

    return (RpLight *)NULL;
}


/*
 *****************************************************************************
 */
static RpLight *
CreateMainLight(RpWorld *world)
{
    RpLight *light;

    /* 
     * Take advantage of any hardware lights if we can...
     */
#if defined(OPENGL_DRVMODEL_H)

    light = RpLightCreate(rpOPENGLLIGHTPOINT);
    if( (RpLight *)NULL != light )
    {
        const RpOpenGLAttenuationParams params = {1.0f, 0.01f, 0.0f};
        
        RpOpenGLLightSetAttenuationParams(light, &params);
    }

#elif defined(D3D7_DRVMODEL_H)
    
    light = RpLightCreate(rpD3D7LIGHTPOINT);
    if( (RpLight *)NULL != light )
    {
        const RpD3D7AttenuationParams params = {1.0f, 0.01f, 0.0f};
        
        RpD3D7LightSetAttenuationParams(light, &params);
    }

#else
    
    light = RpLightCreate(rpLIGHTPOINT);

#endif
    
    if( (RpLight *)NULL != light )
    {
        RwRGBAReal color = {0.7f, 0.7f, 0.7f, 1.0f};
        RwFrame *frame;

        RpLightSetColor(light, &color);
        RpLightSetRadius(light, 100.0f);

        frame = RwFrameCreate();
        if( (RwFrame *)NULL != frame )
        {
            RwV3d pos = {0.0f, 20.0f, 0.0f};
            RwV3d xAxis = {1.0f, 0.0f, 0.0f};

            RwV3dAdd(&pos, &pos, RpWorldGetOrigin(world));
            JumpAttractor = pos;

            RwFrameRotate(frame, &xAxis, 90.0f, rwCOMBINEREPLACE);
            RwFrameTranslate(frame, &pos, rwCOMBINEPOSTCONCAT);

            RpLightSetFrame(light, frame);

            RpWorldAddLight(world, light);

            return light;
        }

        RpLightDestroy(light);
    }

    return (RpLight *)NULL;
}


/*
 *****************************************************************************
 */
static RwCamera *
CreateCamera(RpWorld *world)
{
    RwCamera *camera;

    camera = RdCameraCreate(RsGlobal.maximumWidth, RsGlobal.maximumHeight, TRUE);
    if( (RwCamera *)NULL != camera )
    {
        RwCameraSetNearClipPlane(camera, 0.1f);
        RwCameraSetFarClipPlane(camera, 50.0f);

        RpWorldAddCamera(world, camera);

        CurrentViewIndex = 0;
        CurrentView = PresetViews[CurrentViewIndex];
        CameraSetView(camera, CurrentView);

        return camera;
    }

    return (RwCamera *)NULL;
}


/*
 *****************************************************************************
 */
static RpMaterial *
MaterialSetEnvMap(RpMaterial *material, void *data)
{
    if( EnvMapOn )
    {
        RwFrame *frame;
        RwTexture *texture = (RwTexture *)data;

        frame = RwCameraGetFrame(Camera);

        RpMatFXMaterialSetEffects(material, rpMATFXEFFECTENVMAP);

        RpMatFXMaterialSetupEnvMap(material, texture, frame, FALSE, 0.2f);
    }
    else
    {
        RpMatFXMaterialSetEffects(material, rpMATFXEFFECTNULL);
    }

    return material;
}


static void 
AtomicSetEnvMap(RpAtomic *atomic, RwTexture *texture)
{
    RpGeometryForAllMaterials(RpAtomicGetGeometry(atomic), 
        MaterialSetEnvMap, (void *)texture);

    return;
}


static void 
DiceSetEnvMap(void)
{
    RwInt32 i;

    for(i=0; i<NUMDICE; i++)
    {
        AtomicSetEnvMap(Dice[i].atomic, EnvMap);
    }

    return;
}


/*
 *****************************************************************************
 */
static RpAtomic *
AtomicGetFirst(RpAtomic *atomic, void *data)
{
    *(RpAtomic **)data = atomic;

    return (RpAtomic *)NULL;
}


static RwBool 
CreateDice(RpWorld *world)
{
    RwChar *path;
    RwStream *stream;
    RpClump *clump = (RpClump *)NULL;
    RpAtomic *atomic = (RpAtomic *)NULL;
    RpAtomic *clones[NUMDICE];
    RwTextureFilterMode filterMode;
    RwInt32 i;

    path = RsPathnameCreate(RWSTRING("models/die.dff"));

    /*
     * Read the die clump from a binary format file...
     */
    RsSetModelTexturePath(path);

    stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMREAD, path);
    if( (RwStream *)NULL != stream )
    {
        if( RwStreamFindChunk(stream, rwID_CLUMP, (RwUInt32 *)NULL, (RwUInt32 *)NULL) ) 
        {
            clump = RpClumpStreamRead(stream);
        }

        RwStreamClose(stream, (void *)NULL);
    }

    RsPathnameDestroy(path);

    if( (RpClump *)NULL == clump )
    {
        return FALSE;
    }

    /*
     * Extract the atomic...
     */
    RpClumpForAllAtomics(clump, AtomicGetFirst, &atomic);
    if( (RpAtomic *)NULL == atomic )
    {
        RpClumpDestroy(clump);

        return FALSE;
    }

    filterMode = rwFILTERLINEAR;
    RpGeometryForAllMaterials(RpAtomicGetGeometry(atomic),
        TextureSetFilterMode, (void *)&filterMode);

    EnvMap = RwTextureRead(RWSTRING("gloss"), (RwChar *)NULL);
    RwTextureSetFilterMode(EnvMap, rwFILTERLINEAR);

    /*
     * Create each die by copying this atomic (so we share the geometry)...
     */
    for(i=0; i<NUMDICE; i++)
    {
        clones[i] = (RpAtomic *)NULL;
    }

    for(i=0; i<NUMDICE; i++)
    {
        RwFrame *frame;

        clones[i] = RpAtomicClone(atomic);
        if( (RpAtomic *)NULL == clones[i] )
        {
            break;
        }

        frame = RwFrameCreate();
        if( (RwFrame *)NULL == frame )
        {
            break;
        }

        RpAtomicSetFrame(clones[i], frame);

        RpWorldAddAtomic(world, clones[i]);
    }

    RpClumpDestroy(clump);

    return SetupDice(clones);
}


/*
 *****************************************************************************
 */
static RwBool 
CreateDieFaceRasters(void)
{
    RwChar *imagePath;
    RwImage *image1, *image2;
    RwInt32 i;

    static const RwChar *dieFaceImageNames[6] = 
    { 
        RWSTRING("die1x.png"), 
        RWSTRING("die2x.png"),
        RWSTRING("die3x.png"), 
        RWSTRING("die4x.png"),
        RWSTRING("die5x.png"), 
        RWSTRING("die6x.png")
    };

    imagePath = RsPathnameCreate(RWSTRING("models/textures/"));
    RwImageSetPath(imagePath);
    RsPathnameDestroy(imagePath);

    for(i=0; i<6; i++)
    {
        image1 = RwImageRead(dieFaceImageNames[i]);
        if( (RwImage *)NULL != image1 )
        {
            image2 = RwImageCreateResample(image1, RESAMPLED_SIZE, RESAMPLED_SIZE);

            RwImageDestroy(image1);

            if( (RwImage *)NULL != image2 )
            {
                DieFaceRasters[i] = RwRasterCreate(RESAMPLED_SIZE, RESAMPLED_SIZE, 
                    0, rwRASTERTYPENORMAL);

                if( (RwRaster *)NULL != DieFaceRasters[i] )
                {
                    RwRasterSetFromImage(DieFaceRasters[i], image2);

                    RwImageDestroy(image2);
                }
                else
                {
                    RwImageDestroy(image2);

                    return FALSE;
                }
            }
            else
            {
                return FALSE;
            }
        }
        else
        {
            return FALSE;
        }
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static RwBool 
Initialize(void)
{
    if( RsInitialize() )
    {
        if( 0 == RsGlobal.maximumWidth )
        {
            RsGlobal.maximumWidth = DEFAULT_SCREEN_WIDTH;
        }

        if( 0 == RsGlobal.maximumHeight )
        {
            RsGlobal.maximumHeight = DEFAULT_SCREEN_HEIGHT;
        }

        RsGlobal.appName = RWSTRING("RW3 Dice Demo");

        RsGlobal.maxFPS = 120;

        return TRUE;
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static RwBool 
ResetDiceCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return TRUE;
    }

    InitializeDice();

    return TRUE;
}


static RwBool 
SingleStepCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return TRUE;
    }

    SimulationSingleStep();

    return TRUE;
}


static RwBool 
EnvMapCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return TRUE;
    }

    DiceSetEnvMap();

    return TRUE;
}


void 
ToggleMenuExtension(void)
{
    static RwChar resetLabel[]            = RWSTRING("Reset_R");
    static RwChar singleSteplabel[]       = RWSTRING("Single step_S");
    static RwChar jumpStartLabel[]        = RWSTRING("Jump start_J");
    static RwChar gravityLabel[]          = RWSTRING("Gravity_G");
    static RwChar systemEnergyInfoLabel[] = RWSTRING("System energy_E");
    static RwChar cameraInfoLabel[]       = RWSTRING("Camera info_C");
    static RwChar envMapLabel[]           = RWSTRING("Enviroment-map_M");

    static RwBool menuExtensionOn = FALSE;

    if( menuExtensionOn )
    {
        RdMenuRemoveEntry(&EnvMapOn);

        RdMenuRemoveEntry(&CameraParametersOn);

        RdMenuRemoveEntry(&SystemEnergyInfoOn);

        RdMenuRemoveEntry(&GravityOn);

        RdMenuRemoveEntry(&JumpStartOn);

        RdMenuRemoveEntry(SingleStepCallback);

        RdMenuRemoveEntry(ResetDiceCallback);

        menuExtensionOn = FALSE;
    }
    else
    {
        RdMenuAddEntryTrigger(resetLabel, ResetDiceCallback);
        
        RdMenuAddEntryTrigger(singleSteplabel, SingleStepCallback);
        
        RdMenuAddEntryBool(jumpStartLabel, &JumpStartOn, 
            (RdMenuTriggerCallBack)NULL);
        
        RdMenuAddEntryBool(gravityLabel, &GravityOn, (RdMenuTriggerCallBack)NULL);
        
        RdMenuAddEntryBool(systemEnergyInfoLabel, &SystemEnergyInfoOn, 
            (RdMenuTriggerCallBack)NULL);

        RdMenuAddEntryBool(cameraInfoLabel, &CameraParametersOn, 
            (RdMenuTriggerCallBack)NULL);

        RdMenuAddEntryBool(envMapLabel, &EnvMapOn, EnvMapCallback);

        menuExtensionOn = TRUE;
    }

    return;
}


/*
 *****************************************************************************
 */
static RwBool 
PresetViewCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return TRUE;
    }

    CurrentView = PresetViews[CurrentViewIndex];
    CameraSetView(Camera, CurrentView);

    CurrentDieOn = FALSE;

    return TRUE;
}


static RwBool 
CurrentHFOVCallback(RwBool testEnable)
{
    if( testEnable )
    {
        return TRUE;
    }

    CurrentViewWindow = (RwReal)RwTan(DEG_TO_RAD(CurrentView.hfov * 0.5f));
    RsEventHandler(rsCAMERASIZE, (void *)NULL);

    CurrentDieOn = FALSE;

    return TRUE;
}


static RwBool 
InitializeMenu(void)
{
    static RwChar presetViewLabel[]  = RWSTRING("Preset view_V");
    static RwChar diePreviewLabel[]  = RWSTRING("Die preview_P");
    static RwChar fieldOfViewLabel[] = RWSTRING("Field of view");

    static RwChar fpsLabel[] = RWSTRING("FPS_F");

    static const RwChar *viewNames[] = 
    {  
        RWSTRING("default"), 
        RWSTRING("top"), 
        RWSTRING("closeup"), 
        RWSTRING("surface")
    };

    if( RdMenuOpen(TRUE, &ForegroundColor, &BackgroundColor) )
    {
        RdMenuAddEntryInt(presetViewLabel, &CurrentViewIndex, 
            PresetViewCallback, 0, NumViews-1, 1, viewNames);

        RdMenuAddEntryBool(diePreviewLabel, &DiePreview, (RdMenuTriggerCallBack)NULL);
    
        RdMenuAddEntryReal(fieldOfViewLabel, &CurrentView.hfov, 
            CurrentHFOVCallback, 1.0f, 179.0f, 1.0f);

        RdMenuAddSeparator();

        RdMenuAddEntryBool(fpsLabel, &FPSOn, (RdMenuTriggerCallBack)NULL);

        RdMenuAddSeparator();

        RdMenuSetStatus(rdMENUOFF);

        return TRUE;
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static RwBool 
Initialize3D(void *param)
{
    if( !RsRwInitialize(param) )
    {
        RsErrorMessage(RWSTRING("Error initializing RenderWare."));

        return FALSE;
    }

    Charset = RtCharsetCreate(&ForegroundColor, &BackgroundColor);
    if( (RtCharset *)NULL == Charset )
    {
        RsErrorMessage(RWSTRING("Cannot create raster charset."));
    
        return FALSE;
    }

    World = CreateWorld();
    if( (RpWorld *)NULL == World )
    {
        RsErrorMessage(RWSTRING("Cannot create world."));

        return FALSE;
    }

    AmbientLight = CreateAmbientLight(World);
    if( (RpLight *)NULL == AmbientLight )
    {
        RsErrorMessage(RWSTRING("Cannot create ambient light."));

        return FALSE;
    }

    MainLight = CreateMainLight(World);
    if( (RpLight *)NULL == MainLight )
    {
        RsErrorMessage(RWSTRING("Cannot create main light."));

        return FALSE;
    }

    Camera = CreateCamera(World);
    if( (RwCamera *)NULL == Camera )
    {
        RsErrorMessage(RWSTRING("Cannot create camera."));

        return FALSE;
    }

    Pointer = RdCameraCreateCrossHair();
    if( (RwRaster *)NULL == Pointer )
    {
        RsErrorMessage(RWSTRING("Cannot create camera pointer."));

        return FALSE;
    }

    if( !CreateDice(World) )
    {
        RsErrorMessage(RWSTRING("Error creating dice."));

        return FALSE;
    }

    if( !CreateDieFaceRasters() )
    {
        RsErrorMessage(RWSTRING("Error creating die face rasters."));
        
        return FALSE;
    }

    if( !InitializeMenu() )
    {
        RsErrorMessage(RWSTRING("Error initializing menu."));

        return FALSE;
    }

#ifdef RWMETRICS
    RsMetricsOpen(Camera);
#endif

    return TRUE;
}


/*
 *****************************************************************************
 */
static void 
Terminate3D(void)
{
    RwInt32 i;

#ifdef RWMETRICS
    RsMetricsClose();
#endif

    RdMenuClose();

    for(i=0; i<6; i++)
    {
        if( (RwRaster *)NULL != DieFaceRasters[i] )
        {
            RwRasterDestroy(DieFaceRasters[i]);
        }
    }

    DestroyDice();

    if( (RwRaster *)NULL != Pointer )
    {
        RwRasterDestroy(Pointer);
    }

    if( (RwCamera *)NULL != Camera )
    {
        RpWorldRemoveCamera(World, Camera);

        RdCameraDestroy(Camera);
    }

    if( (RpLight *)NULL != MainLight )
    {
        RwFrame *frame;

        RpWorldRemoveLight(World, MainLight);

        frame = RpLightGetFrame(MainLight);
        RpLightSetFrame(MainLight, (RwFrame *)NULL);
        RwFrameDestroy(frame);

        RpLightDestroy(MainLight);
    }

    if( (RpLight *)NULL != AmbientLight )
    {
        RpWorldRemoveLight(World, AmbientLight);

        RpLightDestroy(AmbientLight);
    }

    if( (RpWorld *)NULL != World )
    {
        RpWorldDestroy(World);
    }

    if( (RtCharset *)NULL != Charset )
    {
        RwRasterDestroy(Charset);
    }

    RsRwTerminate();

    return;
}


/*
 *****************************************************************************
 */
static RwBool 
AttachPlugins(void)
{
    if( !RpWorldPluginAttach() )
    {
        RsErrorMessage(RWSTRING("RpWorldPluginAttach failed."));
        
        return FALSE;
    }

    if( !RpCollisionPluginAttach() )
    {
        RsErrorMessage(RWSTRING("RpCollisionPluginAttach failed."));
        
        return FALSE;
    }

    if( !RpRandomPluginAttach() )
    {
        RsErrorMessage(RWSTRING("RpRandomPluginAttach failed."));
        
        return FALSE;
    }

    if( !RpMatFXPluginAttach() )
    {
        return FALSE;
    }

#ifdef RWLOGO
    if( !RpLogoPluginAttach() )
    {
        RsErrorMessage(RWSTRING("RpLogoPluginAttach failed."));
        
        return FALSE;
    }
#endif

    return TRUE;
}


/*
 *****************************************************************************
 */
static void 
DisplayOnScreenInfo(RwInt32 crw, RwInt32 crh)
{
    RwChar caption[256];
    RtCharsetDesc charsetDesc;

    RtCharsetGetDesc(Charset, &charsetDesc);

    rwstrcpy(caption, RollCaption);
    RtCharsetPrint(Charset, caption, 
        crw - charsetDesc.width * (rwstrlen(caption)+3), 
        crh - charsetDesc.height * 2);

    if( SystemEnergyInfoOn )
    {
        rwsprintf(caption, 
            RWSTRING("Time:%5.2f  TKE:%6.3f  TPE:%6.3f  TE:%6.3f"), 
            RollTime, 
            TotalKineticEnergy, TotalPotentialEnergy, TotalSystemEnergy);

        RtCharsetPrint(Charset, caption, 
            crw - charsetDesc.width * (rwstrlen(caption)+3), 
            crh - charsetDesc.height * 4);
    }

    if( CameraParametersOn )
    {
        rwsprintf(caption, 
            RWSTRING("Camera (Tilt, Turn, Dist): (%6.2f, %6.2f, %6.2f)"), 
            CurrentView.tilt, CurrentView.turn, CurrentView.distance);

        RtCharsetPrint(Charset, caption, 
            crw - charsetDesc.width * (rwstrlen(caption)+3), 
            crh - charsetDesc.height * 5);
    }

    if( FPSOn )
    {
        rwsprintf(caption, RWSTRING("FPS: %03d"), FramesPerSecond);

        RtCharsetPrint(Charset, caption, 
            (crw - charsetDesc.width * (rwstrlen(caption))) / 2,
            charsetDesc.height);
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
Render2DComponents(RwInt32 crw, RwInt32 crh)
{
    /*
     * Display the picked dice...
     */
    if( NumInactiveDice > 0 )
    {
        RwInt32 i;

        for(i=0; i<NumInactiveDice; i++)
        {
            RwRasterRenderFast(DieFaceRasters[PickedDieValues[i]-1], 
                crw - RESAMPLED_SIZE - 10, (i+1)*10 + i*RESAMPLED_SIZE);
        }
    }

    /* 
     * Display the previewed die face, if any, but only if the
     * camera view is not changing...
     */
    if( DiePreview && CurrentDieOn && !(CameraPointing || CameraTranslating))
    {
        RwInt32 x, y;

        x = DiePreviewX + (RESAMPLED_SIZE >> 1);
        y = DiePreviewY + (RESAMPLED_SIZE >> 1);

        if( (x + RESAMPLED_SIZE) > crw )
        {
            x -= RESAMPLED_SIZE << 1;
        }

        if( (y + RESAMPLED_SIZE) > crh )
        {
            y -= RESAMPLED_SIZE << 1;
        }

        RwRasterRenderFast(DieFaceRasters[CurrentDieValue-1], x, y);
    }

    if( CameraPointing || CameraTranslating )
    {
        RwRasterRender(Pointer, 
            (crw - RwRasterGetWidth(Pointer)) >> 1, 
            (crh - RwRasterGetHeight(Pointer)) >> 1);
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
Render(void)
{
    RwRaster *camRas;
    RwInt32 crw, crh;
    RwRGBA clearColor = {0, 0, 0, 255};

    camRas = RwCameraGetRaster(Camera);
    crw = RwRasterGetWidth(camRas);
    crh = RwRasterGetHeight(camRas);

    if( RdMenuGetStatus() == rdHELPMODE )
    {
        clearColor = BackgroundColor;
    }

    RwCameraClear(Camera, &clearColor, rwCAMERACLEARZ | rwCAMERACLEARIMAGE);

    if( RwCameraBeginUpdate(Camera) != (RwCamera *)NULL )
    {
        if( RdMenuGetStatus() != rdHELPMODE )
        {
            RpWorldRender(World);

            DisplayOnScreenInfo(crw, crh);
        }

        RdMenuRender(Camera, (RtCharset *)NULL);

#ifdef RWMETRICS
        RsMetricsRender();
#endif

        RwCameraEndUpdate(Camera);
    }

    if( RdMenuGetStatus() != rdHELPMODE )
    {
        if( RwRasterPushContext(camRas) != (RwRaster *)NULL )
        {
            Render2DComponents(crw, crh);

            RwRasterPopContext();
        }
    }

    /* 
     * Display camera's raster...
     */
    RsCameraShowRaster(Camera);

    FrameCounter++;

    return;
}


/*
 *****************************************************************************
 */
static void 
Idle(void)
{
    RwUInt32 thisTime;

    static RwBool firstCall = TRUE;
    static RwUInt32 lastFrameTime, lastAnimTime;

    if( firstCall )
    {
        lastFrameTime = lastAnimTime = RsTimer();

        firstCall = FALSE;
    }

    thisTime = RsTimer();

    /* 
     * Has a second elapsed since we last updated the FPS...
     */
    if( thisTime > (lastFrameTime + 1000) )
    {
        /* 
         * Capture the frame counter...
         */
        FramesPerSecond = FrameCounter;
        
        /*
         * ...and reset...
         */
        FrameCounter = 0;
        
        lastFrameTime = thisTime;
    }

    RunSimulation(thisTime - lastAnimTime);

    lastAnimTime = thisTime;

    Render();

    return;
}


/*
 *****************************************************************************
 */
RsEventStatus
AppEventHandler(RsEvent event, void *param)
{
    switch( event )
    {
        case rsINITIALIZE:
        {
            return Initialize() ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsCAMERASIZE:
        {
            RdCameraSize(Camera, (RwRect *)param, 
                CurrentViewWindow, DEFAULT_ASPECT_RATIO);

            return rsEVENTPROCESSED;
        }

        case rsRWINITIALIZE:
        {
            return Initialize3D(param) ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsRWTERMINATE:
        {
            Terminate3D();

            return rsEVENTPROCESSED;
        }

        case rsPLUGINATTACH:
        {
            return AttachPlugins() ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsINPUTDEVICEATTACH:
        {
            AttachInputDevices();

            return rsEVENTPROCESSED;
        }

        case rsIDLE:
        {
            Idle();

            return rsEVENTPROCESSED;
        }

        default:
        {
            return rsEVENTNOTPROCESSED;
        }
    }
}

/*
 *****************************************************************************
 */
