
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 1999, 2000 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *
 * dice.c
 *
 * Copyright (C) 1999, 2000 Criterion Technologies.
 *
 * Original author: John Irwin.
 *
 * Purpose: To illustrate collision detection and rigid-body dynamics.
 *
 ****************************************************************************/

#include "rwcore.h"
#include "rpworld.h"
#include "rprandom.h"
#include "rpcollis.h"

#include "skeleton.h"

#include "main.h"
#include "dice.h"
#include "play.h"

#define DIEHALFHEIGHT (0.5f)

#define MAXTIMESTEP (10)
#define FRICTION (0.95f)
#define TIMEOUT (7.0f)

#define MAXCONTACTS (100)


typedef struct
{
    RigidBody *body1;
    RigidBody *body2;

    RwV3d vertexPos;
    RwInt32 vertexIndex;

    RwV3d normal;
    RwV3d velocity;

    RwReal depth;
}
Contact;

typedef enum
{
    csClear,
    csPenetrating,
    csContacting
}
ContactState;


/*
 * This variable defines our contact zone. It extends 
 * from -ContactEpsilon below the collision plane to 
 * +ContactEpsilon above it...
 */
static RwReal ContactEpsilon = 0.005f;

static Contact Contacts[MAXCONTACTS];
static RwInt32 NumOfContacts;
static RwInt32 MaxNumOfContacts;
static RwInt32 MaxNumOfCollisions;

static ContactState ContState;

static RwInt32 SourceStateIndex;
static RwInt32 TargetStateIndex;

static RwV3d Gravity = {0.0f, -30.0f, 0.0f};
static RwV3d Zero = {0.0f, 0.0f, 0.0f};

static RwReal LinDampingCoeffLow  = 0.002f;
static RwReal LinDampingCoeffHigh = 0.04f;
static RwReal AngDampingCoeffLow  = 0.001f;
static RwReal AngDampingCoeffHigh = 0.08f;
static RwReal DampingTime = 5.0f;

static RwReal PEZeroLevel;

static RwV3d DiceOnTablePos[5] = 
{
    {  0.0f, DIEHALFHEIGHT,  0.0f },
    {  1.5f, DIEHALFHEIGHT,  1.5f },
    { -1.5f, DIEHALFHEIGHT,  1.5f },
    {  1.5f, DIEHALFHEIGHT, -1.5f },
    { -1.5f, DIEHALFHEIGHT, -1.5f }
};

static RwV3d InitPos[5] = 
{
    { -4.0f,  2.0f, 6.0f },
    { -6.0f,  2.0f, 4.0f },
    { -4.0f,  2.0f, 4.0f },
    { -6.0f,  2.0f, 6.0f },
    { -5.0f,  4.0f, 5.0f }
};

static RwV3d InitLinVel[5] = 
{
    { 13.0f, 5.0f, -11.0f },
    { 11.0f, 5.0f, -13.0f },
    { 12.0f, 5.0f, -12.0f },
    { 13.0f, 5.0f, -11.0f },
    { 12.0f, 5.0f, -12.0f }
};

static RwMatrix *TempMat1;
static RwMatrix *TempMat2;


RigidBody Dice[NUMDICE];

RwBool SystemEnergyInfoOn = FALSE;
RwBool CameraParametersOn = FALSE;

RwBool GravityOn = TRUE;
RwBool JumpStartOn = FALSE;
RwV3d JumpAttractor;

RwReal RollTime;
RwReal TotalKineticEnergy;
RwReal TotalPotentialEnergy;
RwReal TotalSystemEnergy;
RwReal EnergyThreshold;



/*
 *****************************************************************************
 */
static void 
SimilarityTransform(RwMatrix *out, RwMatrix *in, RwMatrix *orientation)
{
    RwV3d *orientRight, *orientUp, *orientAt;
    RwV3d *tempRight, *tempUp, *tempAt, *tempPos;

    /* 
     * First calculate the transpose of the orientation matrix...
     */
    orientRight = RwMatrixGetRight(orientation);
    orientUp    = RwMatrixGetUp(orientation);
    orientAt    = RwMatrixGetAt(orientation);

    tempRight = RwMatrixGetRight(TempMat1);
    tempUp    = RwMatrixGetUp(TempMat1);
    tempAt    = RwMatrixGetAt(TempMat1);
    tempPos   = RwMatrixGetPos(TempMat1);

    tempRight->x = orientRight->x;
    tempRight->y = orientUp->x;
    tempRight->z = orientAt->x;

    tempUp->x = orientRight->y;
    tempUp->y = orientUp->y;
    tempUp->z = orientAt->y;

    tempAt->x = orientRight->z;
    tempAt->y = orientUp->z;
    tempAt->z = orientAt->z;

    tempPos->x = 0.0f;
    tempPos->y = 0.0f;
    tempPos->z = 0.0f;

    RwMatrixUpdate(TempMat1);
        
    /* 
     * Similarity transformation: 
     * out = transpose(orientation) * in * orientation
     */
    RwMatrixMultiply(TempMat2, TempMat1, in);
    RwMatrixMultiply(out, TempMat2, orientation);

    return;
}


/*
 *****************************************************************************
 */
static void 
RandomVector(RwV3d *vec, RwBool normal)
{
    vec->x = 2.0f * ((RwReal)RpRandom() / (RwUInt32MAXVAL>>1)) - 1.0f;

    vec->y = 2.0f * ((RwReal)RpRandom() / (RwUInt32MAXVAL>>1)) - 1.0f;
    
    vec->z = 2.0f * ((RwReal)RpRandom() / (RwUInt32MAXVAL>>1)) - 1.0f;

    if( normal )
    {
        RwV3dNormalize(vec, vec);
    }

    return;
}


/*
 *****************************************************************************
 */
static RwReal 
RandomUnitReal(void)
{
    return (RwReal)RpRandom() / (RwUInt32MAXVAL>>1);
}


/*
 *****************************************************************************
 */
static RwReal 
TotalEnergy(void)
{
    RigidBody *body;
    State *state;
    RwReal totalKineticEnergy, totalPotentialEnergy;
    RwInt32 i;

    totalKineticEnergy = 0.0f;
    totalPotentialEnergy = 0.0f;

    for(i=0; i<NUMDICE; i++ )
    {
        body = &Dice[i];

        if( body->active )
        {
            state = &body->states[SourceStateIndex];

            totalKineticEnergy += 
                0.5f * RwV3dDotProduct(&state->velocityCM, &state->velocityCM) / body->oneOverMass + 
                0.5f * RwV3dDotProduct(&state->angularMomentum, &state->angularVelocity);

            totalPotentialEnergy += 
                (PEZeroLevel - RwMatrixGetPos(state->matrix)->y) * Gravity.y / body->oneOverMass;
        }
    }

    TotalKineticEnergy = totalKineticEnergy;
    TotalPotentialEnergy = totalPotentialEnergy;
    
    return (totalKineticEnergy + totalPotentialEnergy);
}


/*
 *****************************************************************************
 */
void 
ResetDiceOnTable(void)
{
    RwV3d pos;
    RwFrame *frame;
    RwInt32 dieValue, i;
    RwReal angle;
    State *sourceState;
    RwV3d xAxis = {1.0f, 0.0f, 0.0f};
    RwV3d yAxis = {0.0f, 1.0f, 0.0f};
    RwV3d zAxis = {0.0f, 0.0f, 1.0f};

    for(i=0; i<NUMDICE; i++)
    {
        if( Dice[i].active )
        {
            frame = RpAtomicGetFrame(Dice[i].atomic);

            dieValue = (RwInt32)(6.0f * RandomUnitReal()) + 1;

            switch( dieValue )
            {
                case 1:
                    RwFrameRotate(frame, &xAxis, -90.0f, rwCOMBINEREPLACE);
                    break;

                case 2:
                    RwFrameRotate(frame, &zAxis, -90.0f, rwCOMBINEREPLACE);
                    break;

                case 3:
                    RwFrameRotate(frame, &zAxis, 0.0f, rwCOMBINEREPLACE);
                    break;

                case 4:
                    RwFrameRotate(frame, &zAxis, 180.0f, rwCOMBINEREPLACE);
                    break;

                case 5:
                    RwFrameRotate(frame, &zAxis, 90.0f, rwCOMBINEREPLACE);
                    break;

                case 6:
                    RwFrameRotate(frame, &xAxis, 90.0f, rwCOMBINEREPLACE);
                    break;
            }

            angle = 360.0f * RandomUnitReal();
            RwFrameRotate(frame, &yAxis, angle, rwCOMBINEPOSTCONCAT);

            RwV3dAdd(&pos, RpWorldGetOrigin(World), &DiceOnTablePos[i]);
            RwFrameTranslate(frame, &pos, rwCOMBINEPOSTCONCAT);

            sourceState = &Dice[i].states[SourceStateIndex];
            RwMatrixTransform(sourceState->matrix, 
                RwFrameGetMatrix(frame), rwCOMBINEREPLACE);
        }
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
DampingCoefficients(RwReal *linCoeff, RwReal *angCoeff, RwReal rollTime)
{
    RwReal temp;

    temp = rollTime / DampingTime;

    if( temp < 1.0f )
    {
        temp = temp * temp;
    }
    else if( (temp >= 1.0f) && (temp < 2.0f) )
    {
        temp = temp - 1.0f;
        temp = temp * temp;
        temp = 1.0f - temp;
    }
    else if( temp >= 2.0f )
    {
        temp = 0.0f;
    }

    *linCoeff = LinDampingCoeffLow + 
        (LinDampingCoeffHigh - LinDampingCoeffLow) * temp;
    
    *angCoeff = AngDampingCoeffLow + 
        (AngDampingCoeffHigh - AngDampingCoeffLow) * temp;

    return;
}


/*
 *****************************************************************************
 */
static void 
ComputeForces(RwReal linDampingCoeff, RwReal angDampingCoeff)
{
    RwInt32 i;
    RwV3d tempVec;
    RigidBody *body;
    State *state;

    for(i=0; i<NUMDICE; i++)
    {
        body = &Dice[i];

        if( body->active )
        {
            state  = &body->states[SourceStateIndex];

            /*
             * Clear the forces...
             */
            state->forceCM = Zero;
            state->torque = Zero;

            if( GravityOn )
            {
                /*
                 * Gravity only affects the linear motion, not the angular 
                 * (no torques about the center-of-mass induced by a 
                 * uniform force field)...
                 */
                tempVec = Gravity;

                RwV3dScale(&tempVec, &tempVec, 1.0f / body->oneOverMass);

                RwV3dAdd(&state->forceCM, &state->forceCM, &tempVec);
            }

            /*
             * Damp the linear motion...
             */
            tempVec = state->velocityCM;

            RwV3dScale(&tempVec, &tempVec, -linDampingCoeff);

            RwV3dAdd(&state->forceCM, &state->forceCM, &tempVec);

            /*
             * Damp the angular motion...
             */
            tempVec = state->angularVelocity;

            RwV3dScale(&tempVec, &tempVec, -angDampingCoeff);

            RwV3dAdd(&state->torque, &state->torque, &tempVec);
        }
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
Integrate(RwReal deltaTime)
{
    /*
     * Integration of the equations of motion is accomplished using Euler's
     * method. This is just about as simple as it gets. Probably better to use
     * Runge-Kutta's method of order 4, then we may be able to integrate a
     * full time step in one go, rather than as many steps of MAXTIMESTEP
     * each that fit into the current frame time.
     */
    RwInt32 i;
    RigidBody *body;
    State *sourceState, *targetState;
    RwV3d *right, *up, *at, *pos, tempVec1, tempVec2;

    for(i=0; i<NUMDICE; i++ )
    {
        body = &Dice[i];

        if( body->active )
        {
            sourceState  = &body->states[SourceStateIndex];
            targetState  = &body->states[TargetStateIndex];

            /*
             * Increment the center-of-mass position...
             */
            pos = RwMatrixGetPos(sourceState->matrix);
            RwV3dScale(&tempVec1, &sourceState->velocityCM, deltaTime);
            RwV3dAdd(RwMatrixGetPos(targetState->matrix), pos, &tempVec1);

            /*
             * Increment the center-of-mass velocity...
             */
            RwV3dScale(&tempVec1, &sourceState->forceCM, deltaTime * body->oneOverMass);
            RwV3dAdd(&targetState->velocityCM, &sourceState->velocityCM, &tempVec1);

            /*
             * Increment the orientation by updating the directions of the
             * right, up and at vectors. This makes the matrix slightly
             * non-orthonormal, which is fixed using RwMatrixOrthonormalize.
             * For a speed-up, orthonormalization may be performed less
             * frequently, here it is done every round. Then again, we could
             * use quaternions to circumvent this problem altogether.
             */
            RwV3dScale(&tempVec1, &sourceState->angularVelocity, deltaTime);

            right = RwMatrixGetRight(sourceState->matrix);
            RwV3dCrossProduct(&tempVec2, &tempVec1, right);
            RwV3dAdd(RwMatrixGetRight(targetState->matrix), right, &tempVec2);

            up = RwMatrixGetUp(sourceState->matrix);
            RwV3dCrossProduct(&tempVec2, &tempVec1, up);
            RwV3dAdd(RwMatrixGetUp(targetState->matrix), up, &tempVec2);

            at = RwMatrixGetAt(sourceState->matrix);
            RwV3dCrossProduct(&tempVec2, &tempVec1, at);
            RwV3dAdd(RwMatrixGetAt(targetState->matrix), at, &tempVec2);

            RwMatrixOrthoNormalize(targetState->matrix, targetState->matrix);

            /*
             * Increment the angular momentum (about the center-of-mass)...
             */
            RwV3dScale(&tempVec1, &sourceState->torque, deltaTime);
            RwV3dAdd(&targetState->angularMomentum, 
                &sourceState->angularMomentum, &tempVec1);

            /* 
             * Update the world inertia tensor...
             */
            SimilarityTransform(targetState->invWorldInertiaTensor, 
                body->invObjectInertiaTensor, targetState->matrix);

            /*
             * ...and angular velocity...
             */
            targetState->angularVelocity = targetState->angularMomentum;
            RwV3dTransformVectors(&targetState->angularVelocity, 
                &targetState->angularVelocity, 1, targetState->invWorldInertiaTensor);
        }
    }

    return;
}


/*
 *****************************************************************************
 */
static RwBool 
TriangleVertexContainment(RpCollisionTriangle *triangle, RwV3d *vertex)
{
    RwV3d normal, tempVec;
    RwInt32 i;

    for(i=0; i<3; i++)
    {
        RwV3dSub(&tempVec, triangle->vertices[(i+1)%3], triangle->vertices[i]);
        RwV3dCrossProduct(&normal, &triangle->normal, &tempVec);
        RwV3dSub(&tempVec, vertex, triangle->vertices[i]);

        if( RwV3dDotProduct(&tempVec, &normal ) < 0.0f )
        {
            return FALSE;
        }
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static RpCollisionTriangle * 
WorldCollisionCallback(RpIntersection *intersection, RpWorldSector *sector,
    RpCollisionTriangle *triangle, RwReal distance, void *data)
{
    RigidBody *body;
    State *state;
    RpGeometry *geometry;
    RwV3d *vertices;
    RwInt32 numVertices, j;

    body = (RigidBody *)data;
    state = &body->states[TargetStateIndex];

    geometry = RpAtomicGetGeometry(body->atomic);
    vertices = RpMorphTargetGetVertices(RpGeometryGetMorphTarget(geometry, 0));
    numVertices = RpGeometryGetNumVertices(geometry);

    /* 
     * Search for all vertices from this die that are in
     * the contact zone around this triangle...
     */
    for(j=0; j<numVertices; j++)
    {
        RwReal d;
        RwV3d vertexPos, tempVec;

        /* 
         * Transform the vertex position to world-space...
         */
        RwV3dTransformPoints(&vertexPos, &vertices[j], 1, state->matrix);

        /*
         * Test vertex is contained by planes through each side of
         * the triangle, perpendicular to the triangle plane...
         */
        if( TriangleVertexContainment(triangle, &vertexPos) )
        {
            /* 
             * Calculate the shortest distance of this vertex
             * from the embedding plane...
             */
            RwV3dSub(&tempVec, &vertexPos, &triangle->point);
            d = RwV3dDotProduct(&tempVec, &triangle->normal);

            if( d < -ContactEpsilon )
            {
                /*
                 * This vertex has penetrated beyond the contact zone...
                 */
                ContState = csPenetrating;

                return (RpCollisionTriangle *)NULL;
            }
            else
            {
                if( d < ContactEpsilon )
                {
                    /*
                     * This vertex is in the contact zone...
                     */
                    Contacts[NumOfContacts].body1 = body;
                    Contacts[NumOfContacts].body2 = (RigidBody *)NULL;
                    Contacts[NumOfContacts].vertexPos = vertexPos;
                    Contacts[NumOfContacts].vertexIndex = j;
                    Contacts[NumOfContacts].normal = triangle->normal;
                    Contacts[NumOfContacts].depth = d;

                    NumOfContacts++;

                    if( NumOfContacts >= MAXCONTACTS )
                    {
                        RsErrorMessage(RWSTRING("Too many contacts. Cannot continue."));
                        
                        RsEventHandler(rsQUITAPP, (void *)NULL);
                    }

                    ContState = csContacting;
                }
            }
        }
    }

    return triangle;
}


/*
 *****************************************************************************
 */
static RpAtomic * 
AtomicCollisionCallback(RpIntersection *intersection, RpWorldSector *sector,
    RpAtomic *atomic, RwReal distance, void *data)
{
    RwReal d;

    /*
     * The parameter 'distance' is equal to the distance between the sphere
     * centers divided by the sum of the sphere radii. Hence, if distance = 1,
     * the two spheres are touching.
     */

    /*
     * Calculate the shortest 'distance' between the sphere surfaces...
     */
    d = distance - 1.0f;

    if( d < -ContactEpsilon )
    {
        /*
         * The bounding spheres have intersected too far...
         */
        ContState = csPenetrating;

        return (RpAtomic *)NULL;
    }
    else
    {
        if( d < ContactEpsilon )
        {
            /*
             * We have the bounding spheres in the contact zone...
             */
            RigidBody *body1, *body2;
            State *state1, *state2;
            RwV3d relPosCM;
            RwInt32 i;

            body1 = (RigidBody *)data;
            state1 = &body1->states[TargetStateIndex];

            /*
             * Figure out which rigid-body this atomic represents...
             */
            body2 = (RigidBody *)NULL;
            state2 = (State *)NULL;

            for(i=0; i<NUMDICE; i++)
            {
                if( atomic == Dice[i].atomic )
                {
                    body2 = &Dice[i];
                    state2 = &body2->states[TargetStateIndex];

                    break;
                }
            }

            RwV3dSub(&relPosCM, RwMatrixGetPos(state2->matrix), 
                RwMatrixGetPos(state1->matrix) );

            Contacts[NumOfContacts].body1 = body1;
            Contacts[NumOfContacts].body2 = body2;
            Contacts[NumOfContacts].normal = relPosCM;
            Contacts[NumOfContacts].depth = d;

            NumOfContacts++;

            if( NumOfContacts >= MAXCONTACTS )
            {
                RsErrorMessage(RWSTRING("Too many contacts. Cannot continue."));
                
                RsEventHandler(rsQUITAPP, (void *)NULL);
            }

            ContState = csContacting;
            
        }

        return atomic;
    }
}


/*
 *****************************************************************************
 */
static void 
FindContacts(void)
{
    RpIntersection intersection;
    RigidBody *body;
    State *state;
    RwInt32 i;

    /*
     * Update all active dice atomic-frames so that we can test
     * for collisions properly...
     */
    for(i=0; i<NUMDICE; i++)
    {
        body = &Dice[i];

        if( body->active )
        {
            RwFrame *frame;

            state = &body->states[TargetStateIndex];

            frame = RpAtomicGetFrame(body->atomic);
            RwFrameTransform(frame, state->matrix, rwCOMBINEREPLACE);
        }
    }

    NumOfContacts = 0;
    ContState = csClear;

    intersection.type = rpINTERSECTATOMIC;

    for(i=0; i<NUMDICE; i++)
    {
        body = &Dice[i];

        if( body->active )
        {
            /*
             * Test for contacts between this die and the 
             * world (table) geometry...
             */
            intersection.t.object = body->atomic;

            RpCollisionWorldForAllIntersections(World, &intersection, 
                WorldCollisionCallback, (void *)body);

            if( ContState == csPenetrating )
            {
                return;
            }

            /*
             * Test for contacts between this die's bounding sphere and all 
             * other die bounding spheres. Intersection testing will only occur
             * for those atomics that have the flag rpATOMICCOLLISIONTEST set...
             */
            RpWorldForAllAtomicIntersections(World, &intersection, 
                AtomicCollisionCallback, (void *)body);

            if( ContState == csPenetrating )
            {
                return;
            }
        }
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
ComputeCollisionResponse(Contact *contact)
{
    RigidBody *body1, *body2;
    State *state1, *state2;
    RwV3d relVertPos, impulse, tempVec1, tempVec2;
    RwReal tempReal1, tempReal2;

    if( (RigidBody *)NULL != contact->body2)
    {
        /*
         * We've got a collision between two die's bounding spheres...
         */
        body1 = contact->body1;
        state1 = &body1->states[TargetStateIndex];

        body2 = contact->body2;
        state2 = &body2->states[TargetStateIndex];

        /* 
         * Evaluate the impulse equation for two colliding spheres...
         */
        tempReal1 = (-1.0f - body1->coeffRestitution) * 
            RwV3dDotProduct(&contact->velocity, &contact->normal);

        tempReal2 = (body1->oneOverMass + body2->oneOverMass) * 
            RwV3dDotProduct(&contact->normal, &contact->normal);

        RwV3dScale(&impulse, &contact->normal, tempReal1 / tempReal2);

        /*
         * Apply the impulse to the bounding spheres. This creates a 
         * discontinuous change in the center-of-mass velocities. No
         * change in angular momentum as the impulse is applied through
         * the centre-of-mass. The impulse has dimensions of momentum...
         */
        RwV3dScale(&tempVec1, &impulse, -body1->oneOverMass);
        RwV3dAdd(&state1->velocityCM, &state1->velocityCM, &tempVec1);

        RwV3dScale(&tempVec1, &impulse, body2->oneOverMass);
        RwV3dAdd(&state2->velocityCM, &state2->velocityCM, &tempVec1);
    }
    else
    {
        /*
         * We've got a collision between a die vertex and the (fixed) table...
         */
        body1 = contact->body1;
        state1 = &body1->states[TargetStateIndex];

        /*
         * Vertex position relative to center-of-mass...
         */
        RwV3dSub(&relVertPos, &contact->vertexPos, RwMatrixGetPos(state1->matrix));

        /*
         * Evaluate impulse equation for a vertex colliding with a fixed plane...
         */
        tempReal1 = (-1.0f - body1->coeffRestitution) * 
                    RwV3dDotProduct(&contact->velocity, &contact->normal);

        RwV3dCrossProduct(&tempVec1, &relVertPos, &contact->normal);
        RwV3dTransformVectors(&tempVec1, &tempVec1, 1, state1->invWorldInertiaTensor);
        RwV3dCrossProduct(&tempVec2, &tempVec1, &relVertPos);

        tempReal2 = body1->oneOverMass * 
                    RwV3dDotProduct(&contact->normal, &contact->normal) + 
                    RwV3dDotProduct(&tempVec2, &contact->normal);

        RwV3dScale(&impulse, &contact->normal, tempReal1 / tempReal2);

        /*
         * Apply impulse to the vertex. This creates a discontinuous change of 
         * center-of-mass velocity and angular momentum. The impulse has
         * dimensions of momentum...
         */
        RwV3dScale(&tempVec1, &impulse, body1->oneOverMass);
        RwV3dAdd(&state1->velocityCM, &state1->velocityCM, &tempVec1);

        RwV3dCrossProduct(&tempVec1, &relVertPos, &impulse);
        RwV3dAdd(&state1->angularMomentum, &state1->angularMomentum, &tempVec1);

        /*
         * Update the angular velocity...
         */
        state1->angularVelocity = state1->angularMomentum;
        RwV3dTransformVectors(&state1->angularVelocity, 
            &state1->angularVelocity, 1, state1->invWorldInertiaTensor);

        /*
         * Velocity-dependent friction which reduces the parallel component of the
         * center-of-mass velocity. No such thing as velocity-dependent friction, 
         * but it helps to control the dice sliding on the table top. Calculating
         * true friction forces is a lot more complicated!
         * TempVec1 contains the velocity of the center-of-mass perpendicular to 
         * the colliding surface. TempVec2 contains the velocity parallel to the 
         * surface. Just compose them again at the end to get the resultant center
         * of mass velocity.
         */
        RwV3dScale(&tempVec1, &contact->normal, 
            RwV3dDotProduct(&state1->velocityCM, &contact->normal));

        RwV3dSub(&tempVec2, &state1->velocityCM, &tempVec1);
        RwV3dScale(&tempVec2, &tempVec2, FRICTION);

        RwV3dAdd(&state1->velocityCM, &tempVec2, &tempVec1);
    }

    return;
}


/*
 *****************************************************************************
 */
static RwBool 
Colliding(Contact *contact)
{
    State *state1, *state2;

    if( (RigidBody *)NULL != contact->body2 )
    {
        /*
         * We've got a contact between two die's bounding spheres...
         */
        RwV3d relVelCM;

        state1 = &contact->body1->states[TargetStateIndex];
        state2 = &contact->body2->states[TargetStateIndex];

        RwV3dSub(&relVelCM, &state2->velocityCM, &state1->velocityCM);

        if( RwV3dDotProduct(&relVelCM, &contact->normal) < 0.0f )
        {
            /*
             * Save the relative center-of-mass velocity as we need
             * it to compute the collision response 
             * (saves recomputation)...
             */
            contact->velocity = relVelCM;

            return TRUE;
        }
    }
    else
    {
        /*
         * We've got a contact between a die vertex and the world (table)...
         */
        RwV3d relVertexPos, vertexVelocity, tempVec;

        state1 = &contact->body1->states[TargetStateIndex];

        RwV3dSub(&relVertexPos, &contact->vertexPos, RwMatrixGetPos(state1->matrix));
        RwV3dCrossProduct(&tempVec, &state1->angularVelocity, &relVertexPos);
        RwV3dAdd(&vertexVelocity, &state1->velocityCM, &tempVec);

        if( RwV3dDotProduct(&vertexVelocity, &contact->normal) < 0.0f )
        {
            /*
             * Save the vertex velocity as we need
             * it to compute the collision response
             * (saves recomputation)...
             */
            contact->velocity = vertexVelocity;

            return TRUE;
        }
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static void
AdvanceSimulation(RwReal delta)
{
    RwReal currentTime, targetTime;
    RwReal linDampCoeff, angDampCoeff;

    /*
     * Calculate the linear and angular damping coefficients only 
     * at the beginning of each full time step...
     */
    DampingCoefficients(&linDampCoeff, &angDampCoeff, RollTime);

    currentTime = 0.0f;
    targetTime = delta;

    while( currentTime < delta )
    {
        /*
         * Calculate all forces and torques on the source states...
         */
        ComputeForces(linDampCoeff, angDampCoeff);

        /*
         * Integrate from the source to the target states...
         */
        Integrate(targetTime-currentTime);

        /*
         * Find contacts between target states and world (table)...
         */
        FindContacts();

        if( ContState == csPenetrating )
        {
            /*
             * Integrated too far, so subdivide and advance simulation
             * from original source state over new time step...
             */
            targetTime = (currentTime + targetTime) * 0.5f;

            if( RwRealAbs(currentTime-targetTime) < 0.00001f )
            {
                RsErrorMessage(RWSTRING("Cannot find contact time."));

                currentTime = delta;

                InitializeDice();
            }
        }
        else
        {
            /*
             * We may have some dice in the contact zone...
             */
            if( ContState == csContacting )
            {
                /* 
                 * At least one vertex or bounding sphere is in the
                 * contact zone. Cycle through the array of contacts
                 * and determine which are real collisions by checking if
                 * the bodies in question are moving towards each other
                 * and, if so, apply an impulse. Repeat until there are
                 * no more real collisions. After this all contacting
                 * entities should be moving away from the contact zone...
                 */
                RwBool hadCollision;
                RwInt32 numOfCollisions, i;

                numOfCollisions = 0;

                do
                {
                    hadCollision = FALSE;

                    for(i=0; i<NumOfContacts; i++)
                    {
                        if( Colliding(&Contacts[i]) )
                        {
                            ComputeCollisionResponse(&Contacts[i]);

                            numOfCollisions++;

                            if( numOfCollisions > 1000 )
                            {
                                RsErrorMessage(RWSTRING("Cannot resolve collisions."));

                                DiceRolling = FALSE;

                                hadCollision = FALSE;

                                break;
                            }

                            hadCollision = TRUE;
                        }
                    }
                }
                while( hadCollision == TRUE );

                if( NumOfContacts > MaxNumOfContacts )
                {
                    MaxNumOfContacts = NumOfContacts;
                }

                if( numOfCollisions > MaxNumOfCollisions )
                {
                    MaxNumOfCollisions = numOfCollisions;
                }
            }

            /*
             * If we get here everything is clear: no penetrations
             * and all contacting vertices or bounding spheres are 
             * moving away from the contact zone...
             */
            currentTime = targetTime;
            targetTime = delta;

            /*
             * Swap the source and target states ready for the 
             * next simulation step...or render...
             */
            SourceStateIndex = SourceStateIndex == 1 ? 0 : 1;
            TargetStateIndex = TargetStateIndex == 1 ? 0 : 1;

        }
    }

    RollTime += delta;

    return;
}


/*
 *****************************************************************************
 */
static void 
TestEndOfRoll(void)
{
    TotalSystemEnergy = TotalEnergy();

    if( TotalSystemEnergy < EnergyThreshold )
    {
        /*
         * The active dice have all settled on the table as 
         * much as they can, so turn off the simulation...
         */
        DiceRolling = FALSE;

        /*
         * If that was the last roll, display the die values
         * left on the table, put all the dice back on the 
         * table...
         */
        if( 2 == RollNumber )
        {
            RwInt32 i;

            GatherRemainingDieValues();

            for(i=0; i<NUMDICE; i++)
            {
                Dice[i].active = TRUE;

                RpAtomicSetFlags(Dice[i].atomic, rpATOMICRENDER);
            }

            ResetDiceOnTable();

            CurrentDieOn = FALSE;

            rwstrcpy(RollCaption, RWSTRING("Game Over"));
        }
        else
        {
            rwstrcat(RollCaption, RWSTRING("over"));
        }
    }

    if( (TotalPotentialEnergy < 0.0f) || (RollTime > TIMEOUT) )
    {
        /*
         * A die has either jumped off the table or is sitting
         * on the table rim...
         */
        rwstrcpy(RollCaption, RWSTRING("Invalid roll. Try again!"));

        RollNumber--;

        DiceRolling = FALSE;

        ResetDiceOnTable();
    }


    return;
}


/*
 *****************************************************************************
 */
void 
RunSimulation(RwUInt32 milliseconds)
{
    if( DiceRolling )
    {
        while( milliseconds > 0 )
        {
            RwUInt32 deltaTime;

            deltaTime = milliseconds;

            if( deltaTime > MAXTIMESTEP )
            {
                deltaTime = MAXTIMESTEP;
            }

            AdvanceSimulation(deltaTime * 0.001f);

            milliseconds -= deltaTime;
        }

        /* 
         * See if everything has settled down yet...
         */
        TestEndOfRoll();
    }

    return;
}


/*
 *****************************************************************************
 */
void
SimulationSingleStep(void)
{
    DiceRolling = FALSE;

    AdvanceSimulation(MAXTIMESTEP * 0.001f);
    
    TotalSystemEnergy = TotalEnergy();

    return;
}


/*
 *****************************************************************************
 */
void 
InitializeDice(void)
{
    RwInt32 i;
    RwV3d tempVec;
    RigidBody *body;
    State *sourceState;

    SourceStateIndex = 0;
    TargetStateIndex = 1;

    for(i=0; i<NUMDICE; i++)
    {
        body = &Dice[i];

        if( body->active )
        {
            sourceState = &body->states[SourceStateIndex];

            if( JumpStartOn )
            {
                RwV3d unit;
                RwReal speed;
                RwSphere *atomicSphere;
                RwV3d yAxis = {0.0f, 1.0f, 0.0f};

                /*
                 * Unit vector from die to light source...
                 */
                RwV3dSub(&unit, &JumpAttractor, RwMatrixGetPos(sourceState->matrix));
                RwV3dNormalize(&unit, &unit);

                /*
                 * Random speed between 10 -> 20...
                 */
                speed = 10.0f + 10.0f * RandomUnitReal();

                /*
                 * Initial velocity...
                 */
                RwV3dScale(&sourceState->velocityCM, &unit, speed);

                /*
                 * Move the die off the table enough to prevent any intial
                 * collisions with the table triangles. Only move it vertically
                 * so that it doesn't create an intersection with a neighbouring
                 * die...
                 */
                atomicSphere = RpAtomicGetBoundingSphere(body->atomic);
                RwV3dScale(&tempVec, &yAxis, atomicSphere->radius);
                RwMatrixTranslate(sourceState->matrix, &tempVec, rwCOMBINEPOSTCONCAT);
            }
            else
            {
                RwReal angle;

                RwV3dAdd(&tempVec, &InitPos[i], RpWorldGetOrigin(World));
                RwMatrixTranslate(sourceState->matrix, &tempVec, rwCOMBINEREPLACE);

                sourceState->velocityCM = InitLinVel[i];

                RandomVector(&tempVec, FALSE);
                angle = 360.0f * RandomUnitReal();

                RwMatrixRotate(sourceState->matrix, &tempVec, angle, rwCOMBINEPRECONCAT);
            }

            /*
             * Now we can calculate the world-space inertia tensor...
             */
            SimilarityTransform(sourceState->worldInertiaTensor, 
                body->objectInertiaTensor, sourceState->matrix);

            SimilarityTransform(sourceState->invWorldInertiaTensor,
                body->invObjectInertiaTensor, sourceState->matrix);

            /*
             * ...and initialize the angular velocity and angular momentum...
             */
            RandomVector(&tempVec, TRUE);
            RwV3dScale(&sourceState->angularVelocity, &tempVec, 2.0f * rwPI);
            /*sourceState->angularVelocity = Zero;*/

            sourceState->angularMomentum = sourceState->angularVelocity;
            RwV3dTransformVectors(&sourceState->angularMomentum, 
                &sourceState->angularMomentum, 1, sourceState->worldInertiaTensor);
        }
    }

    /*
     * Calculate the initial total kinetic and potential energy of all the dice ...
     */
    TotalSystemEnergy = TotalEnergy();

    RollTime = 0.0f;

    return;
}


/*
 *****************************************************************************
 */
RwBool 
SetupDice(RpAtomic *dice[])
{
    RwBool success = TRUE;
    RwInt32 i;

    /*
     * First check that we have all the atomics, initialize as we go...
     */
    for(i=0; i<NUMDICE; i++)
    {
        Dice[i].atomic = dice[i];

        if( (RpAtomic *)NULL == dice[i] )
        {
            success = FALSE;
        }
    }

    if( !success )
    {
        return FALSE;
    }

    for(i=0; i<NUMDICE; i++)
    {
        RigidBody *body;
        RwV3d *right, *up, *at;
        RwReal dieMass;

        body = &Dice[i];

        dieMass = 0.05f;
        body->oneOverMass = 1.0f / dieMass;
        body->coeffRestitution = 0.7f;

        /*
         * Create and initialize the object-space inertia tensor...
         */
        body->objectInertiaTensor = RwMatrixCreate();

        /*
         * ...calculate the die's object-space inertia tensor
         * using an approximation (though good one) which 
         * assumes the die is a perfect cube of the given size...
         */
        RwMatrixSetIdentity(body->objectInertiaTensor);

        right = RwMatrixGetRight(body->objectInertiaTensor);
        up = RwMatrixGetUp(body->objectInertiaTensor);
        at = RwMatrixGetAt(body->objectInertiaTensor);

        right->x = (2.0f * dieMass * DIEHALFHEIGHT * DIEHALFHEIGHT) / 3.0f;
        up->y = at->z = right->x;

        RwMatrixUpdate(body->objectInertiaTensor);

        /*
         * ...and the inverse object-space inertia tensor...
         */
        body->invObjectInertiaTensor = RwMatrixCreate();

        RwMatrixSetIdentity(body->invObjectInertiaTensor);

        right = RwMatrixGetRight(body->invObjectInertiaTensor);
        up = RwMatrixGetUp(body->invObjectInertiaTensor);
        at = RwMatrixGetAt(body->invObjectInertiaTensor);

        right->x = 3.0f / (2.0f * dieMass * DIEHALFHEIGHT * DIEHALFHEIGHT);
        up->y = at->z = right->x;

        RwMatrixUpdate(body->invObjectInertiaTensor);

        /*
         * Create all state matrices...
         */
        body->states[0].matrix = RwMatrixCreate();

        body->states[0].worldInertiaTensor = RwMatrixCreate();

        body->states[0].invWorldInertiaTensor = RwMatrixCreate();

        body->states[1].matrix = RwMatrixCreate();

        body->states[1].worldInertiaTensor = RwMatrixCreate();

        body->states[1].invWorldInertiaTensor = RwMatrixCreate();

        body->active = TRUE;
    }

    EnergyThreshold = MINENERGY;

    MaxNumOfContacts = MaxNumOfCollisions = 0;

    NumInactiveDice = 0;

    ResetDiceOnTable();

    TempMat1 = RwMatrixCreate();
    TempMat2 = RwMatrixCreate();

    /*
     * The plane of the dice table is known to pass through 
     * the origin and is perpendicular to the y-axis. The zero
     * of potential energy corresponds to a die lying flat on
     * the table, so negative values mean a die has escaped 
     * the table and is falling towards oblivion...
     */
    PEZeroLevel = RpWorldGetOrigin(World)->y + DIEHALFHEIGHT;

    return TRUE;
}


/*
 *****************************************************************************
 */
void
DestroyDice(void)
{
    RwInt32 i;

    for(i=0; i<NUMDICE; i++)
    {
        RigidBody *body;

        body = &Dice[i];

        if( (RpAtomic *)NULL != body->atomic )
        {
            RwFrame *frame;
            RpWorld *world;

            frame = RpAtomicGetFrame(body->atomic);
            if( (RwFrame *)NULL != frame )
            {
                RpAtomicSetFrame(body->atomic, (RwFrame *)NULL);
                RwFrameDestroy(frame);
            }

            world = RpAtomicGetWorld(body->atomic);
            if( (RpWorld *)NULL != world )
            {
                RpWorldRemoveAtomic(world, body->atomic);
            }

            RpAtomicDestroy(body->atomic);

            if( (RwMatrix *)NULL != body->objectInertiaTensor )
            {
                RwMatrixDestroy(body->objectInertiaTensor);
            }

            if( (RwMatrix *)NULL != body->invObjectInertiaTensor )
            {
                RwMatrixDestroy(body->invObjectInertiaTensor);
            }

            if( (RwMatrix *)NULL != body->states[0].matrix )
            {
                RwMatrixDestroy(body->states[0].matrix);
            }

            if( (RwMatrix *)NULL != body->states[0].worldInertiaTensor )
            {
                RwMatrixDestroy(body->states[0].worldInertiaTensor);
            }

            if( (RwMatrix *)NULL != body->states[0].invWorldInertiaTensor )
            {
                RwMatrixDestroy(body->states[0].invWorldInertiaTensor);
            }

            if( (RwMatrix *)NULL != body->states[1].matrix )
            {
                RwMatrixDestroy(body->states[1].matrix);
            }

            if( (RwMatrix *)NULL != body->states[1].worldInertiaTensor )
            {
                RwMatrixDestroy(body->states[1].worldInertiaTensor);
            }

            if( (RwMatrix *)NULL != body->states[1].invWorldInertiaTensor )
            {
                RwMatrixDestroy(body->states[1].invWorldInertiaTensor);
            }
        }
    }

    if( (RwMatrix *)NULL != TempMat1 )
    {
        RwMatrixDestroy(TempMat1);
    }

    if( (RwMatrix *)NULL != TempMat2 )
    {
        RwMatrixDestroy(TempMat2);
    }

    return;
}

/*
 *****************************************************************************
 */
